<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require '../db.php';
require '../helpers.php';

// Check committee access (session is already started in header/db)
if (!isset($_SESSION['committee_id'])) {
    header('Location: ../login.php');
    exit;
}

require_committee();

$committeeId   = (int)($_SESSION['committee_id'] ?? 0);
$committeeName = (string)($_SESSION['committee_name'] ?? '');
$canAll        = (bool)($_SESSION['committee_can_all'] ?? false);
$scopeCollege  = isset($_SESSION['committee_college_id']) ? (int)$_SESSION['committee_college_id'] : null;

// Initialize variables
$students = [];

// Check if filters were submitted
$filters_applied = isset($_GET['filter_applied']) && $_GET['filter_applied'] === '1';

// Get filter parameters - only use them if filters were applied
if ($filters_applied) {
    $filter_semester_id = $_GET['filter_semester'] ?? null;
    $filter_college_id = $_GET['filter_college'] ?? null;
    $filter_section_id = $_GET['filter_section'] ?? null;
    $filter_program_id = $_GET['filter_program'] ?? null;
    $filter_batch_id = $_GET['filter_batch'] ?? null;
    $filter_exam_type = $_GET['filter_exam_type'] ?? 'midterm';
} else {
    // Don't apply filters by default
    $filter_semester_id = null;
    $filter_college_id = null;
    $filter_section_id = null;
    $filter_program_id = null;
    $filter_batch_id = null;
    $filter_exam_type = 'midterm';
}

// Get semester name for display
$semester_name = '';
if ($filter_semester_id) {
    $semester_sql = "SELECT name, code FROM semesters WHERE id = ? AND deleted_at IS NULL";
    $st_sem = $mysqli1->prepare($semester_sql);
    $st_sem->bind_param("i", $filter_semester_id);
    $st_sem->execute();
    $semester_result = $st_sem->get_result();
    $semester_data = $semester_result->fetch_assoc();
    if ($semester_data) {
        $semester_name = $semester_data['name'] . ' (' . $semester_data['code'] . ')';
    }
    $st_sem->close();
}

// Build query to get students
if ($filters_applied) {
    $sql = "SELECT s.student_id, s.student_name, s.student_name_eng, 
                   sec.name as section_name, 
                   b.value as batch_name,
                   p.name as program_name,
                   c.name as college_name
            FROM students s 
            LEFT JOIN sections sec ON sec.id = s.section_id
            LEFT JOIN batches b ON b.id = s.batch_id
            LEFT JOIN programs p ON p.id = s.program_id
            LEFT JOIN colleges c ON c.id = s.college_id
            WHERE 1=1";

    $params = [];
    $types = '';

    // Apply scope restrictions
    if (!$canAll && $scopeCollege !== null) {
        $sql .= " AND s.college_id = ? ";
        $params[] = $scopeCollege;
        $types .= 'i';
        // Auto-set the college filter if user has scope restriction
        if (!$filter_college_id) {
            $filter_college_id = $scopeCollege;
        }
    } elseif (!$canAll && $scopeCollege === null) {
        $sql .= " AND 1=0 ";
    }

    // Apply filters
    if ($filter_college_id && !empty($filter_college_id)) {
        $sql .= " AND s.college_id = ? ";
        $params[] = $filter_college_id;
        $types .= 'i';
    }

    if ($filter_section_id && !empty($filter_section_id)) {
        $sql .= " AND s.section_id = ? ";
        $params[] = $filter_section_id;
        $types .= 'i';
    }

    if ($filter_program_id && !empty($filter_program_id)) {
        $sql .= " AND s.program_id = ? ";
        $params[] = $filter_program_id;
        $types .= 'i';
    }

    if ($filter_batch_id && !empty($filter_batch_id)) {
        $sql .= " AND s.batch_id = ? ";
        $params[] = $filter_batch_id;
        $types .= 'i';
    }

    $sql .= " ORDER BY s.student_name";

    $st = $mysqli1->prepare($sql);
    if ($types && !empty($params)) {
        $st->bind_param($types, ...$params);
    }
    $st->execute();
    $result = $st->get_result();
    $students = $result->fetch_all(MYSQLI_ASSOC);
    $st->close();
} else {
    $students = [];
}

// Get filter options
$colleges_stmt = $mysqli1->prepare("SELECT id, name FROM colleges WHERE deleted_at IS NULL ORDER BY name");
$colleges_stmt->execute();
$colleges = $colleges_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$colleges_stmt->close();

// Get sections for the selected college
$sections = [];
if ($filter_college_id && !empty($filter_college_id)) {
    $sections_sql = "
        SELECT DISTINCT s.id, s.name, s.parent
        FROM sections s 
        WHERE s.parent = ? AND s.deleted_at IS NULL 
        ORDER BY s.name
    ";
    $sections_stmt = $mysqli1->prepare($sections_sql);
    $sections_stmt->bind_param("i", $filter_college_id);
    $sections_stmt->execute();
    $sections = $sections_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $sections_stmt->close();
}

// Get programs based on selected college and section
$programs = [];
if ($filter_college_id && !empty($filter_college_id)) {
    $programs_where = "p.deleted_at IS NULL";
    $programs_params = [];
    $programs_types = "";
    
    if ($filter_section_id && !empty($filter_section_id)) {
        $programs_where .= " AND st.section_id = ?";
        $programs_types .= "i";
        $programs_params[] = $filter_section_id;
    }
    
    $programs_where .= " AND st.college_id = ?";
    $programs_types .= "i";
    $programs_params[] = $filter_college_id;
    
    $programs_sql = "
        SELECT DISTINCT p.id, p.name 
        FROM programs p 
        INNER JOIN students st ON p.id = st.program_id 
        WHERE {$programs_where} 
        ORDER BY p.name
    ";
    
    $programs_stmt = $mysqli1->prepare($programs_sql);
    if ($programs_types) {
        $programs_stmt->bind_param($programs_types, ...$programs_params);
    }
    $programs_stmt->execute();
    $programs = $programs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $programs_stmt->close();
} else {
    $programs_stmt = $mysqli1->prepare("SELECT id, name FROM programs WHERE deleted_at IS NULL ORDER BY name");
    $programs_stmt->execute();
    $programs = $programs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $programs_stmt->close();
}

$batches_stmt = $mysqli1->prepare("SELECT id, value FROM batches WHERE deleted_at IS NULL ORDER BY id DESC");
$batches_stmt->execute();
$batches = $batches_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$batches_stmt->close();

// Get semesters
$semesters_stmt = $mysqli1->prepare("SELECT id, name, code FROM semesters WHERE deleted_at IS NULL ORDER BY id DESC");
$semesters_stmt->execute();
$semesters = $semesters_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$semesters_stmt->close();

// Get filter names for display
$college_name = "";
if ($filter_college_id) {
    foreach ($colleges as $college) {
        if ($college['id'] == $filter_college_id) {
            $college_name = $college['name'];
            break;
        }
    }
}

$section_name = "";
if ($filter_section_id && !empty($sections)) {
    foreach ($sections as $section) {
        if ($section['id'] == $filter_section_id) {
            $section_name = $section['name'];
            break;
        }
    }
}

$program_name = "";
if ($filter_program_id) {
    foreach ($programs as $program) {
        if ($program['id'] == $filter_program_id) {
            $program_name = $program['name'];
            break;
        }
    }
}

$batch_name = "";
if ($filter_batch_id) {
    foreach ($batches as $batch) {
        if ($batch['id'] == $filter_batch_id) {
            $batch_name = $batch['value'];
            break;
        }
    }
}

include '../header.php';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تقرير حضور الامتحان</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body, h1,h2,h3,h4,h5,h6, p, a, button, label { 
            font-family: 'Cairo', system-ui, -apple-system, sans-serif !important; 
        }
        .card-arabic {
            border: 1px solid #e5e7eb;
            border-radius: 16px;
            background: #fff;
            box-shadow: 0 10px 24px rgba(0,0,0,.06);
            margin-bottom: 20px;
        }
        .filter-icon {
            width: 40px;
            height: 40px;
            background: #3b82f6;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.1rem;
        }
        .filter-status {
            font-size: 0.85rem;
            padding: 0.3rem 0.6rem;
            border-radius: 6px;
        }
        .filter-status.applied {
            background-color: #d1fae5;
            color: #065f46;
            border: 1px solid #a7f3d0;
        }
        .filter-status.not-applied {
            background-color: #fee2e2;
            color: #991b1b;
            border: 1px solid #fecaca;
        }
        .info-message {
            background-color: #f0f9ff;
            border: 1px solid #bae6fd;
            color: #0369a1;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        .summary-card {
            background: white;
            border-radius: 12px;
            padding: 15px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.06);
            border: 1px solid #e0e0e0;
            text-align: center;
        }
        .summary-card .number {
            font-size: 2rem;
            font-weight: 800;
            margin-bottom: 5px;
        }
        .summary-card .label {
            font-size: 0.9rem;
            color: #6c757d;
        }
        .student-table th {
            background-color: #f8f9fa;
            font-weight: 600;
        }
        .badge-custom {
            padding: 5px 10px;
            border-radius: 20px;
            font-weight: 500;
        }
        .form-select, .form-control {
            border-radius: 8px;
            border: 1px solid #ced4da;
        }
        .form-select:focus, .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
        }
        .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #2563eb);
            border: none;
            border-radius: 8px;
            padding: 8px 20px;
        }
        .btn-primary:hover {
            background: linear-gradient(135deg, #2563eb, #1d4ed8);
        }
        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.06);
        }
        .chip {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
        }
        .chip.success { background: #d4edda; color: #155724; }
        .chip.info { background: #d1ecf1; color: #0c5460; }
        .chip.warning { background: #fff3cd; color: #856404; }
        .chip.primary { background: #dbeafe; color: #1e3a8a; }
        .chip.secondary { background: #e9ecef; color: #495057; }
        .chip.blue { background: #d0ebff; color: #0c63e4; border: 1px solid #a5d8ff; }
        .chip.green { background: #d3f9d8; color: #2b8a3e; border: 1px solid #b2f2bb; }
        
        /* Print styles */
        @media print {
            body { 
                font-size: 12px !important; 
                direction: rtl;
                padding: 0;
                margin: 0;
            }
            
            .no-print { 
                display: none !important; 
            }
            
            .print-only { 
                display: block !important; 
            }
            
            .print-header {
                text-align: center;
                margin-bottom: 20px;
                padding-bottom: 15px;
            }
            
            .print-info {
                font-size: 11px;
                color: #666;
                margin-bottom: 10px;
            }
            
            .print-table {
                width: 100%;
                border-collapse: collapse;
                font-size: 10px;
            }
            
            .print-table th {
                background-color: #f0f0f0 !important;
                border: 1px solid #000 !important;
                padding: 6px 4px !important;
                font-weight: bold !important;
                text-align: center !important;
            }
            
            .print-table td {
                border: 1px solid #000 !important;
                padding: 5px 4px !important;
                text-align: center !important;
            }
            
            .print-table .text-right {
                text-align: right !important;
                padding-right: 10px !important;
            }
            
            .print-footer {
                margin-top: 20px;
                padding-top: 10px;
                border-top: 1px solid #000;
                font-size: 10px;
                text-align: center;
                color: #666;
            }
            
            /* Hide all elements except print content */
            body > *:not(.print-container) {
                display: none !important;
            }
        }
        
        .print-container {
            display: none;
        }
        
        .banner-img {
            max-width: 100%;
            height: auto;
            margin-bottom: 15px;
        }
        
        .student-name-cell {
            text-align: right !important;
            direction: rtl !important;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            padding-right: 15px !important;
        }
    </style>
</head>
<body>
    <div class="container py-4">
        <!-- Header -->
        <div class="d-flex align-items-center mb-4 no-print">
            <h3 class="mb-0 fw-bold">تقرير حضور الامتحان</h3>
            <div class="ms-auto">
                <span class="filter-status <?= $filters_applied ? 'applied' : 'not-applied' ?> me-2">
                    <i class="fas fa-<?= $filters_applied ? 'check-circle' : 'exclamation-triangle' ?> me-1"></i>
                    <?= $filters_applied ? 'تم تطبيق التصفية' : 'لم يتم تطبيق أي تصفية' ?>
                </span>
                <a class="btn btn-outline-secondary" href="../committee_dashboard.php">
                    <i class="fas fa-arrow-right me-2"></i>العودة للوحة التحكم
                </a>
            </div>
        </div>

        <?php if (!$filters_applied): ?>
            <div class="info-message no-print">
                <i class="fas fa-info-circle me-2"></i>
                <strong>تنبيه:</strong> لم يتم تطبيق أي تصفية. الرجاء اختيار عوامل التصفية ثم النقر على "تطبيق التصفية" لعرض الطلاب.
            </div>
        <?php endif; ?>

        <!-- Filter Card -->
        <div class="card card-arabic p-4 mb-4 no-print">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <div class="d-flex align-items-center mb-3">
                        <div class="filter-icon me-2">
                            <i class="fas fa-filter"></i>
                        </div>
                        <h6 class="fw-bold mb-0 text-dark">تصفية الطلاب لحضور الامتحان</h6>
                        <div class="ms-3">
                            <span class="badge bg-light text-dark border">
                                <i class="fas fa-sliders-h me-1"></i>
                                <?= $filters_applied ? 'تصفية مطبقة' : 'في انتظار التطبيق' ?>
                            </span>
                        </div>
                    </div>
                    
                    <form method="get" class="row g-3 align-items-end" id="filterForm">
                        <!-- Hidden field to indicate filters were applied -->
                        <input type="hidden" name="filter_applied" value="1">
                        
                        <!-- College Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-university me-1"></i>
                                    الكلية
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_college" id="filter_college" onchange="loadSections(this.value)">
                                        <option value="">اختر الكلية</option>
                                        <?php foreach ($colleges as $college): ?>
                                            <?php 
                                            // Don't show college filter if user has scope restriction
                                            if (!$canAll && $scopeCollege !== null && $college['id'] != $scopeCollege) {
                                                continue;
                                            }
                                            ?>
                                            <option value="<?= $college['id'] ?>" <?= ($filter_college_id == $college['id']) ? 'selected' : '' ?>>
                                                <?= e($college['name']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-building"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Section Filter (Dependent on College) -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-layer-group me-1"></i>
                                    القسم
                                </label>
                                <div class="input-group input-group-sm position-relative">
                                    <select class="form-control border-end-0" name="filter_section" id="filter_section" <?= !$filter_college_id ? 'disabled' : '' ?>>
                                        <option value="">جميع الأقسام</option>
                                        <?php if ($filter_college_id && !empty($sections)): ?>
                                            <?php foreach ($sections as $section): ?>
                                                <option value="<?= $section['id'] ?>" <?= ($filter_section_id == $section['id']) ? 'selected' : '' ?>>
                                                    <?= e($section['name']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-building"></i>
                                    </span>
                                </div>
                                <?php if (!$filter_college_id): ?>
                                    <small class="text-muted">الرجاء اختيار كلية أولاً</small>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Program Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-graduation-cap me-1"></i>
                                    البرنامج
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_program" id="filter_program">
                                        <option value="">جميع البرامج</option>
                                        <?php foreach ($programs as $program): ?>
                                            <option value="<?= $program['id'] ?>" <?= ($filter_program_id == $program['id']) ? 'selected' : '' ?>>
                                                <?= e($program['name']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-book"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Batch Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-users me-1"></i>
                                    الدفعة
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_batch">
                                        <option value="">جميع الدفعات</option>
                                        <?php foreach ($batches as $batch): ?>
                                            <option value="<?= $batch['id'] ?>" <?= ($filter_batch_id == $batch['id']) ? 'selected' : '' ?>>
                                                <?= e($batch['value']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-calendar-alt"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Semester Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-calendar-alt me-1"></i>
                                    الفصل الدراسي
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_semester">
                                        <option value="">جميع الفصول</option>
                                        <?php foreach ($semesters as $semester): ?>
                                            <option value="<?= $semester['id'] ?>" <?= ($filter_semester_id == $semester['id']) ? 'selected' : '' ?>>
                                                <?= e($semester['name'] . ' (' . $semester['code'] . ')') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-calendar"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Exam Type Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-file-alt me-1"></i>
                                    نوع الامتحان
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_exam_type">
                                        <option value="midterm" <?= ($filter_exam_type == 'midterm') ? 'selected' : '' ?>>منتصف الفصل</option>
                                        <option value="final" <?= ($filter_exam_type == 'final') ? 'selected' : '' ?>>نهاية الفصل</option>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-file-alt"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="col-12">
                            <div class="d-flex gap-2 pt-2">
                                <button type="submit" class="btn btn-primary btn-sm px-3">
                                    <i class="fas fa-filter me-1"></i>
                                    تطبيق التصفية
                                </button>
                                <a href="exam_attendance.php" class="btn btn-outline-secondary btn-sm px-3">
                                    <i class="fas fa-sync-alt me-1"></i>
                                    إعادة تعيين
                                </a>
                                <?php if ($filters_applied): ?>
                                    <span class="badge bg-success align-self-center px-3 py-2">
                                        <i class="fas fa-check me-1"></i>
                                        تم تطبيق التصفية
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Filter Summary -->
                <div class="col-md-4">
                    <div class="bg-light rounded p-3 h-100">
                        <div class="d-flex align-items-center mb-2">
                            <i class="fas fa-info-circle me-2 <?= $filters_applied ? 'text-primary' : 'text-secondary' ?>"></i>
                            <span class="fw-semibold">ملخص التصفية</span>
                        </div>
                        <div class="small">
                            <?php if ($filters_applied): ?>
                                <div class="mb-2">
                                    <strong>عدد الطلاب:</strong> 
                                    <span class="badge bg-info"><?= number_format(count($students)) ?></span>
                                </div>
                                
                                <!-- Show selected exam type -->
                                <div class="mb-1">
                                    <span class="fw-semibold">نوع الامتحان:</span>
                                    <span class="badge bg-primary bg-opacity-10 text-primary">
                                        <?= $filter_exam_type == 'midterm' ? 'منتصف الفصل' : 'نهاية الفصل' ?>
                                    </span>
                                </div>
                                
                                <!-- Show selected semester -->
                                <?php if ($filter_semester_id && $semester_name): ?>
                                    <div class="mb-1">
                                        <span class="fw-semibold">الفصل الدراسي:</span>
                                        <span class="badge bg-secondary bg-opacity-10 text-secondary">
                                            <?= e($semester_name) ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                                
                                <!-- Show selected college and section -->
                                <?php if ($filter_college_id && $college_name): ?>
                                    <div class="mb-1">
                                        <span class="fw-semibold">الكلية:</span>
                                        <span class="badge bg-info bg-opacity-10 text-info">
                                            <?= e($college_name) ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($filter_section_id && $section_name && $filter_college_id && !empty($sections)): ?>
                                    <div class="mb-1">
                                        <span class="fw-semibold">القسم:</span>
                                        <span class="badge bg-success bg-opacity-10 text-success">
                                            <?= e($section_name) ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($filter_program_id && $program_name): ?>
                                    <div class="mb-1">
                                        <span class="fw-semibold">البرنامج:</span>
                                        <span class="badge bg-warning bg-opacity-10 text-warning">
                                            <?= e($program_name) ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($filter_batch_id && $batch_name): ?>
                                    <div class="mb-1">
                                        <span class="fw-semibold">الدفعة:</span>
                                        <span class="badge bg-dark bg-opacity-10 text-dark">
                                            <?= e($batch_name) ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="mb-1 mt-3">شروط التصفية:</div>
                                <ul class="small mb-0 ps-3">
                                    <li>جميع الطلاب بناءً على عوامل التصفية</li>
                                    <li>الطلاب غير المحذوفين فقط</li>
                                    <?php if (!$canAll && $scopeCollege !== null): ?>
                                        <li>نطاق الكلية محدود</li>
                                    <?php endif; ?>
                                </ul>
                            <?php else: ?>
                                <div class="text-center py-3">
                                    <i class="fas fa-filter fa-2x text-muted mb-3"></i>
                                    <p class="text-muted mb-0">لم يتم تطبيق أي تصفية بعد</p>
                                    <p class="text-muted small">الرجاء اختيار عوامل التصفية ثم النقر على "تطبيق التصفية"</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Report Content -->
        <?php if ($filters_applied): ?>
            <!-- Exam Information Card -->
            <div class="card card-arabic p-4 mb-4 no-print">
                <div class="d-flex align-items-center mb-3">
                    <div class="filter-icon me-2" style="background: #10b981;">
                        <i class="fas fa-info-circle"></i>
                    </div>
                    <h6 class="fw-bold mb-0 text-dark">معلومات الامتحان</h6>
                </div>
                
                <form id="examInfoForm" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label fw-semibold text-muted mb-2">
                            <i class="fas fa-door-open me-1"></i>
                            اسم القاعة
                        </label>
                        <div class="input-group input-group-sm">
                            <input type="text" class="form-control border-end-0" id="room_name" placeholder="أدخل اسم القاعة">
                            <span class="input-group-text bg-light border-start-0">
                                <i class="fas fa-building"></i>
                            </span>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <label class="form-label fw-semibold text-muted mb-2">
                            <i class="fas fa-book me-1"></i>
                            اسم المادة
                        </label>
                        <div class="input-group input-group-sm">
                            <input type="text" class="form-control border-end-0" id="exam_name" placeholder="أدخل اسم المادة">
                            <span class="input-group-text bg-light border-start-0">
                                <i class="fas fa-book-open"></i>
                            </span>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <label class="form-label fw-semibold text-muted mb-2">
                            <i class="fas fa-calendar-day me-1"></i>
                            تاريخ الامتحان
                        </label>
                        <div class="input-group input-group-sm">
                            <input type="date" class="form-control border-end-0" id="exam_date">
                            <span class="input-group-text bg-light border-start-0">
                                <i class="fas fa-calendar"></i>
                            </span>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <label class="form-label fw-semibold text-muted mb-2">
                            <i class="fas fa-clock me-1"></i>
                            وقت الامتحان
                        </label>
                        <div class="input-group input-group-sm">
                            <input type="time" class="form-control border-end-0" id="exam_time">
                            <span class="input-group-text bg-light border-start-0">
                                <i class="fas fa-clock"></i>
                            </span>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Summary Cards -->
            <div class="row mb-4 no-print">
                <div class="col-md-3">
                    <div class="summary-card">
                        <div class="number text-info"><?= number_format(count($students)) ?></div>
                        <div class="label">إجمالي الطلاب</div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="summary-card">
                        <div class="number text-success">
                            <?= number_format(count($students) * 4) ?>
                        </div>
                        <div class="label">الحقول المطلوبة</div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="summary-card">
                        <div class="number text-warning">
                            <?= number_format(count($students)) ?>
                        </div>
                        <div class="label">توقيعات مطلوبة</div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="summary-card">
                        <div class="number text-primary">
                            <?= $filter_exam_type == 'midterm' ? 'منتصف الفصل' : 'نهاية الفصل' ?>
                        </div>
                        <div class="label">نوع الامتحان</div>
                    </div>
                </div>
            </div>

            <!-- Export and Print Options -->
            <div class="row mb-4 no-print">
                <div class="col-12">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <div class="text-muted">
                                <i class="fas fa-info-circle me-1"></i>
                                آخر تحديث: <?= date('Y-m-d H:i:s') ?>
                                <?php if ($program_name): ?>
                                    | <i class="fas fa-graduation-cap me-1"></i>
                                    البرنامج: <?= e($program_name) ?>
                                <?php endif; ?>
                                <?php if ($semester_name): ?>
                                    | <i class="fas fa-calendar-alt me-1"></i>
                                    الفصل: <?= e($semester_name) ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="d-flex gap-2">
                            <button class="btn btn-success btn-sm" onclick="exportToExcel()">
                                <i class="fas fa-file-excel me-1"></i>
                                تصدير لإكسل
                            </button>
                            <button class="btn btn-danger btn-sm" onclick="printReportPDF()">
                                <i class="fas fa-print me-1"></i>
                                طباعة PDF
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Students Table -->
            <div class="card card-arabic p-4">
                <div class="d-flex align-items-center mb-3">
                    <h6 class="fw-bold mb-0">
                        <i class="fas fa-file-signature me-2 text-info"></i>
                        سجل حضور الامتحان
                        <span class="chip primary ms-2">
                            <i class="fas fa-file-alt me-1"></i>
                            <?= $filter_exam_type == 'midterm' ? 'منتصف الفصل' : 'نهاية الفصل' ?>
                        </span>
                        <?php if ($program_name): ?>
                            <span class="chip warning ms-2">
                                <i class="fas fa-graduation-cap me-1"></i>
                                <?= e($program_name) ?>
                            </span>
                        <?php endif; ?>
                    </h6>
                    <div class="ms-auto">
                        <span class="badge bg-info">
                            <i class="fas fa-users me-1"></i>
                            <?= number_format(count($students)) ?> طالب
                        </span>
                    </div>
                </div>

                <div class="table-responsive">
                    <table class="table table-striped student-table" id="studentsTable">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>الرقم الجامعي</th>
                                <th>اسم الطالب</th>
                                <th class="text-center">الدرجة</th>
                                <th class="text-center">أعمال السنة</th>
                                <th class="text-center">المجموع</th>
                                <th class="text-center">توقيع الطالب</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($students)): ?>
                                <tr>
                                    <td colspan="7" class="text-center text-muted py-4">
                                        <i class="fas fa-info-circle me-2"></i>
                                        لا توجد بيانات للعرض بناءً على عوامل التصفية المحددة
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($students as $index => $student): ?>
                                    <tr>
                                        <td class="fw-bold"><?= $index + 1 ?></td>
                                        <td>
                                            <code><?= e($student['student_id']) ?></code>
                                        </td>
                                        <td class="text-right student-name-cell">
                                            <?= e($student['student_name'] ?: $student['student_name_eng']) ?>
                                        </td>
                                        <td class="text-center" style="height: 45px; background-color: #f8f9fa; vertical-align: middle;">
                                            <!-- Empty for grade -->
                                        </td>
                                        <td class="text-center" style="height: 45px; background-color: #f8f9fa; vertical-align: middle;">
                                            <!-- Empty for midterm remarks -->
                                        </td>
                                        <td class="text-center" style="height: 45px; background-color: #f8f9fa; vertical-align: middle;">
                                            <!-- Empty for final remarks -->
                                        </td>
                                        <td class="text-center" style="height: 45px; background-color: #f8f9fa; vertical-align: middle;">
                                            <!-- Empty for signature -->
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- Hidden Print Container -->
        <div class="print-container" id="printContainer">
            <div class="print-header">
                <!-- Banner Image -->
                <img src="../img/banar3.jpg" alt="Banner" class="banner-img" onerror="this.style.display='none'">
                
                <h2>تقرير حضور الامتحان</h2>
                <div class="print-info" id="printExamInfo">
                    <!-- Exam information will be filled by JavaScript -->
                </div>
            </div>
            
            <table class="print-table" id="printTable">
                <thead>
                    <tr>
                        <th width="4%">#</th>
                        <th width="12%">الرقم الجامعي</th>
                        <th width="20%">اسم الطالب</th>
                        <th width="8%">توقيع الطالب</th>
                        <th width="8%">الحضور</th>
                        <th width="8%">تمارين</th>
                        <th width="8%">عملي</th>
                        <th width="8%">نصف فترة</th>
                        <th width="8%">نهائي</th>
                        <th width="8%">المجموع</th>
                        <th width="8%">التقدير</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($students)): ?>
                        <tr>
                            <td colspan="11" class="text-center">
                                لا توجد بيانات للعرض
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($students as $index => $student): ?>
                            <tr>
                                <td><?= $index + 1 ?></td>
                                <td><?= e($student['student_id']) ?></td>
                                <td class="text-right">
                                    <?= e($student['student_name'] ?: $student['student_name_eng']) ?>
                                </td>
                                <td></td>
                                <td></td>
                                <td></td>
                                <td></td>
                                <td></td>
                                <td></td>
                                <td></td>
                                <td></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <div class="print-footer">
                تم إنشاء التقرير تلقائياً من نظام إدارة القبول والتسجيل<br>
                الصفحة 1 من 1
            </div>
        </div>
    </div>

    <script>
        // Simple section loading - reload page when college changes
        function loadSections(collegeId) {
            const sectionSelect = document.getElementById('filter_section');
            const programSelect = document.getElementById('filter_program');
            
            // Reset section and program selects
            sectionSelect.innerHTML = '<option value="">جميع الأقسام</option>';
            programSelect.innerHTML = '<option value="">جميع البرامج</option>';
            
            if (!collegeId || collegeId === '') {
                sectionSelect.disabled = true;
                return;
            }
            
            // Create a form and submit it to reload the page
            const form = document.createElement('form');
            form.method = 'get';
            form.style.display = 'none';
            
            // Add current exam type
            const currentExamType = "<?= $filter_exam_type ?>";
            if (currentExamType) {
                const examTypeInput = document.createElement('input');
                examTypeInput.type = 'hidden';
                examTypeInput.name = 'filter_exam_type';
                examTypeInput.value = currentExamType;
                form.appendChild(examTypeInput);
            }
            
            // Add current semester
            const currentSemester = "<?= $filter_semester_id ?>";
            if (currentSemester) {
                const semesterInput = document.createElement('input');
                semesterInput.type = 'hidden';
                semesterInput.name = 'filter_semester';
                semesterInput.value = currentSemester;
                form.appendChild(semesterInput);
            }
            
            // Add college parameter
            const collegeInput = document.createElement('input');
                collegeInput.type = 'hidden';
            collegeInput.name = 'filter_college';
            collegeInput.value = collegeId;
            form.appendChild(collegeInput);
            
            // Add filter_applied parameter
            const appliedInput = document.createElement('input');
            appliedInput.type = 'hidden';
            appliedInput.name = 'filter_applied';
            appliedInput.value = '1';
            form.appendChild(appliedInput);
            
            // Add current section if exists
            const currentSection = "<?= $filter_section_id ?>";
            if (currentSection) {
                const sectionInput = document.createElement('input');
                sectionInput.type = 'hidden';
                sectionInput.name = 'filter_section';
                sectionInput.value = currentSection;
                form.appendChild(sectionInput);
            }
            
            // Add current program if exists
            const currentProgram = "<?= $filter_program_id ?>";
            if (currentProgram) {
                const programInput = document.createElement('input');
                programInput.type = 'hidden';
                programInput.name = 'filter_program';
                programInput.value = currentProgram;
                form.appendChild(programInput);
            }
            
            // Add current batch if exists
            const currentBatch = "<?= $filter_batch_id ?>";
            if (currentBatch) {
                const batchInput = document.createElement('input');
                batchInput.type = 'hidden';
                batchInput.name = 'filter_batch';
                batchInput.value = currentBatch;
                form.appendChild(batchInput);
            }
            
            document.body.appendChild(form);
            form.submit();
        }
        
        // Export to Excel function
        function exportToExcel() {
            if (!<?= $filters_applied ? 'true' : 'false' ?>) {
                alert('الرجاء تطبيق التصفية أولاً لعرض الطلاب');
                return;
            }
            
            // Get exam information
            const roomName = document.getElementById('room_name').value || 'غير محدد';
            const examName = document.getElementById('exam_name').value || 'غير محدد';
            const examDate = document.getElementById('exam_date').value || 'غير محدد';
            const examTime = document.getElementById('exam_time').value || 'غير محدد';
            const examType = "<?= $filter_exam_type == 'midterm' ? 'منتصف الفصل' : 'نهاية الفصل' ?>";
            const semesterName = "<?= e($semester_name) ?>";
            
            // Get the table
            const table = document.getElementById('studentsTable');
            
            if (!table) {
                alert('لا توجد بيانات للتصدير');
                return;
            }
            
            // Create a new workbook
            const wb = XLSX.utils.book_new();
            
            // Create worksheet from table
            const ws = XLSX.utils.table_to_sheet(table);
            
            // Add exam information as first rows
            const examInfo = [
                [`سجل حضور الامتحان - ${examType}`],
                [`اسم القاعة: ${roomName} | اسم المادة: ${examName}`],
                [`تاريخ الامتحان: ${examDate} | وقت الامتحان: ${examTime}`],
                [`الفصل الدراسي: ${semesterName}`],
                [''], // Empty row for spacing
            ];
            
            // Convert table to array and add exam info at the beginning
            const tableArray = XLSX.utils.sheet_to_json(ws, {header: 1});
            const combinedArray = [...examInfo, ...tableArray];
            
            // Create new worksheet with combined data
            const newWs = XLSX.utils.aoa_to_sheet(combinedArray);
            
            // Set column widths
            const wscols = [
                {wch: 5},   // #
                {wch: 15},  // Student ID
                {wch: 25},  // Student Name
                {wch: 12},  // Grade
                {wch: 15},  // Year Work
                {wch: 12},  // Total
                {wch: 15},  // Signature
            ];
            newWs['!cols'] = wscols;
            
            // Add worksheet to workbook
            XLSX.utils.book_append_sheet(wb, newWs, "سجل حضور الامتحان");
            
            // Generate filename
            let filename = "سجل_حضور_الامتحان";
            <?php if ($program_name): ?>
                filename += "_<?= e($program_name) ?>";
            <?php endif; ?>
            <?php if ($semester_name): ?>
                filename += "_<?= e($semester_name) ?>";
            <?php endif; ?>
            filename += "_<?= date('Y-m-d') ?>.xlsx";
            
            // Clean filename (remove special characters)
            filename = filename.replace(/[<>:"/\\|?*]+/g, '_');
            
            // Save the file
            XLSX.writeFile(wb, filename);
        }
        
        // PDF Print function
        function printReportPDF() {
            if (!<?= $filters_applied ? 'true' : 'false' ?>) {
                alert('الرجاء تطبيق التصفية أولاً لعرض الطلاب');
                return;
            }
            
            if (<?= count($students) ?> === 0) {
                alert('لا توجد بيانات للطباعة');
                return;
            }
            
            // Get exam information
            const roomName = document.getElementById('room_name').value || 'غير محدد';
            const examName = document.getElementById('exam_name').value || 'غير محدد';
            const examDate = document.getElementById('exam_date').value || 'غير محدد';
            const examTime = document.getElementById('exam_time').value || 'غير محدد';
            const examType = "<?= $filter_exam_type == 'midterm' ? 'منتصف الفصل' : 'نهاية الفصل' ?>";
            const semesterName = "<?= e($semester_name) ?>";
            
            // Create a new window for printing
            const printWindow = window.open('', '_blank', 'width=1000,height=800');
            
            // Get the print container HTML
            const printContainer = document.getElementById('printContainer').outerHTML;
            
            // Update exam info in print container
            const printInfo = `
                نوع الامتحان: ${examType}<br>
                الفصل الدراسي: ${semesterName}<br>
                اسم القاعة: ${roomName} | اسم المادة: ${examName}<br>
                تاريخ الامتحان: ${examDate} | وقت الامتحان: ${examTime}<br>
                <?php if ($college_name): ?>الكلية: <?= e($college_name) ?><br><?php endif; ?>
                <?php if ($program_name): ?>البرنامج: <?= e($program_name) ?><br><?php endif; ?>
            `;
            
            // Replace the exam info div content
            const updatedPrintContainer = printContainer.replace(
                '<div class="print-info" id="printExamInfo">',
                `<div class="print-info" id="printExamInfo">${printInfo}`
            );
            
            // Create the print content
            const printContent = `
                <!DOCTYPE html>
                <html dir="rtl">
                <head>
                    <meta charset="UTF-8">
                    <title>تقرير حضور الامتحان</title>
                    <style>
                        @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap');
                        
                        * {
                            font-family: 'Cairo', sans-serif !important;
                            box-sizing: border-box;
                        }
                        
                        body {
                            margin: 0;
                            padding: 15mm;
                            direction: rtl;
                            background: white;
                            font-size: 12px;
                        }
                        
                        .print-header {
                            text-align: center;
                            margin-bottom: 20px;
                            padding-bottom: 10px;
                        }
                        
                        .banner-img {
                            max-width: 100%;
                            height: auto;
                            margin-bottom: 15px;
                        }
                        
                        .print-header h2 {
                            font-size: 20px;
                            margin: 5px 0;
                            font-weight: bold;
                            color: #000;
                        }
                        
                        .print-info {
                            font-size: 11px;
                            color: #666;
                            margin-top: 10px;
                            line-height: 1.6;
                        }
                        
                        .print-table {
                            width: 100%;
                            border-collapse: collapse;
                            font-size: 10px;
                            margin-top: 15px;
                        }
                        
                        .print-table th {
                            background-color: #f0f0f0;
                            border: 1px solid #000;
                            padding: 8px 5px;
                            font-weight: bold;
                            text-align: center;
                            font-size: 11px;
                        }
                        
                        .print-table td {
                            border: 1px solid #000;
                            padding: 6px 5px;
                            text-align: center;
                        }
                        
                        .print-table .text-right {
                            text-align: right;
                            padding-right: 10px;
                        }
                        
                        .print-footer {
                            margin-top: 20px;
                            padding-top: 10px;
                            border-top: 1px solid #666;
                            font-size: 10px;
                            text-align: center;
                            color: #666;
                            line-height: 1.6;
                        }
                        
                        @media print {
                            @page {
                                size: A4 landscape;
                                margin: 15mm;
                            }
                            
                            body { 
                                margin: 0;
                                padding: 0;
                            }
                            
                            .print-table {
                                font-size: 9px;
                            }
                            
                            .print-table th, 
                            .print-table td {
                                padding: 5px 3px;
                            }
                        }
                    </style>
                </head>
                <body>
                    ${updatedPrintContainer}
                </body>
                </html>
            `;
            
            printWindow.document.write(printContent);
            printWindow.document.close();
            
            // Trigger print after content is loaded
            printWindow.onload = function() {
                printWindow.focus();
                setTimeout(function() {
                    printWindow.print();
                    // Optional: Close after printing
                    // printWindow.close();
                }, 500);
            };
        }
        
        // Initialize when page loads
        document.addEventListener('DOMContentLoaded', function() {
            const collegeSelect = document.getElementById('filter_college');
            const sectionSelect = document.getElementById('filter_section');
            
            // Enable/disable section select based on college selection
            if (collegeSelect && sectionSelect) {
                if (!collegeSelect.value) {
                    sectionSelect.disabled = true;
                } else {
                    sectionSelect.disabled = false;
                }
            }
            
            // Load saved exam info from localStorage
            const savedRoom = localStorage.getItem('exam_room');
            const savedExam = localStorage.getItem('exam_name');
            const savedDate = localStorage.getItem('exam_date');
            const savedTime = localStorage.getItem('exam_time');
            
            if (savedRoom) document.getElementById('room_name').value = savedRoom;
            if (savedExam) document.getElementById('exam_name').value = savedExam;
            if (savedDate) document.getElementById('exam_date').value = savedDate;
            if (savedTime) document.getElementById('exam_time').value = savedTime;
            
            // Save exam info when they change
            document.getElementById('room_name').addEventListener('input', function() {
                localStorage.setItem('exam_room', this.value);
            });
            document.getElementById('exam_name').addEventListener('input', function() {
                localStorage.setItem('exam_name', this.value);
            });
            document.getElementById('exam_date').addEventListener('input', function() {
                localStorage.setItem('exam_date', this.value);
            });
            document.getElementById('exam_time').addEventListener('input', function() {
                localStorage.setItem('exam_time', this.value);
            });
        });
    </script>
    
    <!-- Load SheetJS library for Excel export -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
</body>
</html>