<?php
// /admission3333/reports/unregistered_students.php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../helpers.php';

// Check committee access
if (!isset($_SESSION['committee_id'])) {
    header('Location: ../login.php');
    exit;
}

require_committee();

$committeeId   = (int)($_SESSION['committee_id'] ?? 0);
$committeeName = (string)($_SESSION['committee_name'] ?? '');
$canAll        = (bool)($_SESSION['committee_can_all'] ?? false);
$scopeCollege  = isset($_SESSION['committee_college_id']) ? (int)$_SESSION['committee_college_id'] : null;

// Initialize variables
$students = [];

// Check if filters were submitted
$filters_applied = isset($_GET['filter_applied']) && $_GET['filter_applied'] === '1';

// Get filter parameters - only use them if filters were applied
if ($filters_applied) {
    $filter_college_id = $_GET['filter_college'] ?? null;
    $filter_section_id = $_GET['filter_section'] ?? null;
    $filter_program_id = $_GET['filter_program'] ?? null;
    $filter_batch_id = $_GET['filter_batch'] ?? null;
    $filter_semester_id = $_GET['filter_semester'] ?? null;
} else {
    // Don't apply filters by default
    $filter_college_id = null;
    $filter_section_id = null;
    $filter_program_id = null;
    $filter_batch_id = null;
    $filter_semester_id = null;
}

// Validate that semester is required when filters are applied
if ($filters_applied && empty($filter_semester_id)) {
    $error_message = "الرجاء اختيار فصل دراسي لعرض الطلاب غير المسجلين";
    $filters_applied = false; // Don't show students without semester
}

// Build WHERE clause for filtering students
$where = " WHERE 1=1 ";
$types = ''; 
$params = [];

// Apply scope restrictions
if (!$canAll && $scopeCollege !== null) {
    $where .= " AND s.college_id = ? ";
    $types .= 'i'; 
    $params[] = $scopeCollege;
    // Auto-set the college filter if user has scope restriction
    if (!$filter_college_id) {
        $filter_college_id = $scopeCollege;
    }
} elseif (!$canAll && $scopeCollege === null) {
    $where .= " AND 1=0 ";
}

// Apply filters ONLY if they were applied by user
if ($filters_applied) {
    if ($filter_college_id && !empty($filter_college_id)) {
        $where .= " AND s.college_id = ? ";
        $types .= 'i'; 
        $params[] = $filter_college_id;
    }

    if ($filter_section_id && !empty($filter_section_id)) {
        $where .= " AND s.section_id = ? ";
        $types .= 'i'; 
        $params[] = $filter_section_id;
    }

    if ($filter_program_id && !empty($filter_program_id)) {
        $where .= " AND s.program_id = ? ";
        $types .= 'i'; 
        $params[] = $filter_program_id;
    }

    if ($filter_batch_id && !empty($filter_batch_id)) {
        $where .= " AND s.batch_id = ? ";
        $types .= 'i'; 
        $params[] = $filter_batch_id;
    }
}

// Get filtered students ONLY if filters were applied and semester is selected
if ($filters_applied && !empty($filter_semester_id)) {
    // Get semester name
    $semester_stmt = $mysqli1->prepare("SELECT name, code FROM semesters WHERE id = ?");
    $semester_stmt->bind_param("i", $filter_semester_id);
    $semester_stmt->execute();
    $semester_result = $semester_stmt->get_result();
    $semester_data = $semester_result->fetch_assoc();
    $semester_name = $semester_data['name'] ?? '';
    $semester_code = $semester_data['code'] ?? '';
    $semester_stmt->close();
    
    // Query for unregistered students in the selected semester
    $sqlStudents = "
        SELECT 
            s.student_id,
            s.student_name,
            s.student_name_eng,
            s.college_id,
            s.section_id,
            s.program_id,
            s.batch_id,
            c.name AS college_name,
            sec.name AS section_name,
            p.name AS program_name,
            b.value AS batch_name
        FROM students s
        LEFT JOIN colleges c ON s.college_id = c.id
        LEFT JOIN sections sec ON s.section_id = sec.id
        LEFT JOIN programs p ON s.program_id = p.id
        LEFT JOIN batches b ON s.batch_id = b.id
        {$where}
        AND s.student_id NOT IN (
            SELECT DISTINCT student_id 
            FROM registrations 
            WHERE semester_id = ? AND is_paid = 1 AND deleted_at IS NULL
        )
        ORDER BY s.student_name
        LIMIT 500
    ";

    // Add semester parameter
    $types .= 'i';
    $params[] = $filter_semester_id;
    
    $students_stmt = $mysqli1->prepare($sqlStudents);
    if ($types) {
        $students_stmt->bind_param($types, ...$params);
    }
    $students_stmt->execute();
    $students = $students_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $students_stmt->close();
} else {
    // No students to show initially
    $students = [];
}

// Get filter options
$colleges_stmt = $mysqli1->prepare("SELECT id, name FROM colleges WHERE deleted_at IS NULL ORDER BY name");
$colleges_stmt->execute();
$colleges = $colleges_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$colleges_stmt->close();

// Get sections for the selected college
$sections = [];
if ($filter_college_id && !empty($filter_college_id)) {
    $sections_sql = "
        SELECT DISTINCT s.id, s.name, s.parent
        FROM sections s 
        WHERE s.parent = ? AND s.deleted_at IS NULL 
        ORDER BY s.name
    ";
    $sections_stmt = $mysqli1->prepare($sections_sql);
    $sections_stmt->bind_param("i", $filter_college_id);
    $sections_stmt->execute();
    $sections = $sections_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $sections_stmt->close();
}

// Get programs based on selected college and section
$programs = [];
if ($filter_college_id && !empty($filter_college_id)) {
    $programs_where = "p.deleted_at IS NULL";
    $programs_params = [];
    $programs_types = "";
    
    if ($filter_section_id && !empty($filter_section_id)) {
        $programs_where .= " AND st.section_id = ?";
        $programs_types .= "i";
        $programs_params[] = $filter_section_id;
    }
    
    $programs_where .= " AND st.college_id = ?";
    $programs_types .= "i";
    $programs_params[] = $filter_college_id;
    
    $programs_sql = "
        SELECT DISTINCT p.id, p.name 
        FROM programs p 
        INNER JOIN students st ON p.id = st.program_id 
        WHERE {$programs_where} 
        ORDER BY p.name
    ";
    
    $programs_stmt = $mysqli1->prepare($programs_sql);
    if ($programs_types) {
        $programs_stmt->bind_param($programs_types, ...$programs_params);
    }
    $programs_stmt->execute();
    $programs = $programs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $programs_stmt->close();
} else {
    $programs_stmt = $mysqli1->prepare("SELECT id, name FROM programs WHERE deleted_at IS NULL ORDER BY name");
    $programs_stmt->execute();
    $programs = $programs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $programs_stmt->close();
}

$batches_stmt = $mysqli1->prepare("SELECT id, value FROM batches WHERE deleted_at IS NULL ORDER BY id DESC");
$batches_stmt->execute();
$batches = $batches_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$batches_stmt->close();

// Get semesters
$semesters_stmt = $mysqli1->prepare("SELECT id, name, code FROM semesters WHERE deleted_at IS NULL ORDER BY id DESC");
$semesters_stmt->execute();
$semesters = $semesters_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$semesters_stmt->close();

// Get filter names for display
$college_name = "";
if ($filter_college_id) {
    foreach ($colleges as $college) {
        if ($college['id'] == $filter_college_id) {
            $college_name = $college['name'];
            break;
        }
    }
}

$section_name = "";
if ($filter_section_id && !empty($sections)) {
    foreach ($sections as $section) {
        if ($section['id'] == $filter_section_id) {
            $section_name = $section['name'];
            break;
        }
    }
}

$program_name = "";
if ($filter_program_id) {
    foreach ($programs as $program) {
        if ($program['id'] == $filter_program_id) {
            $program_name = $program['name'];
            break;
        }
    }
}

$batch_name = "";
if ($filter_batch_id) {
    foreach ($batches as $batch) {
        if ($batch['id'] == $filter_batch_id) {
            $batch_name = $batch['value'];
            break;
        }
    }
}

$semester_name = "";
if ($filter_semester_id && isset($semester_name)) {
    $semester_name = $semester_name . ' (' . $semester_code . ')';
}

// Include header with correct path
include __DIR__ . '/../header.php';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>الطلاب غير المسجلين - تقرير</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body, h1,h2,h3,h4,h5,h6, p, a, button, label { 
            font-family: 'Cairo', system-ui, -apple-system, sans-serif !important; 
        }
        .card-arabic {
            border: 1px solid #e5e7eb;
            border-radius: 16px;
            background: #fff;
            box-shadow: 0 10px 24px rgba(0,0,0,.06);
            margin-bottom: 20px;
        }
        .filter-icon {
            width: 40px;
            height: 40px;
            background: #ef4444;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.1rem;
        }
        .filter-status {
            font-size: 0.85rem;
            padding: 0.3rem 0.6rem;
            border-radius: 6px;
        }
        .filter-status.applied {
            background-color: #fee2e2;
            color: #991b1b;
            border: 1px solid #fecaca;
        }
        .filter-status.not-applied {
            background-color: #fef3c7;
            color: #92400e;
            border: 1px solid #fde68a;
        }
        .info-message {
            background-color: #fef3c7;
            border: 1px solid #fbbf24;
            color: #92400e;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        .warning-message {
            background-color: #fee2e2;
            border: 1px solid #f87171;
            color: #991b1b;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        .summary-card {
            background: white;
            border-radius: 12px;
            padding: 15px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.06);
            border: 1px solid #e0e0e0;
            text-align: center;
        }
        .summary-card .number {
            font-size: 2rem;
            font-weight: 800;
            margin-bottom: 5px;
        }
        .summary-card .label {
            font-size: 0.9rem;
            color: #6c757d;
        }
        .student-table th {
            background-color: #f8f9fa;
            font-weight: 600;
        }
        .badge-custom {
            padding: 5px 10px;
            border-radius: 20px;
            font-weight: 500;
        }
        .form-select, .form-control {
            border-radius: 8px;
            border: 1px solid #ced4da;
        }
        .form-select:focus, .form-control:focus {
            border-color: #ef4444;
            box-shadow: 0 0 0 0.2rem rgba(239, 68, 68, 0.25);
        }
        .btn-primary {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            border: none;
            border-radius: 8px;
            padding: 8px 20px;
        }
        .btn-primary:hover {
            background: linear-gradient(135deg, #dc2626, #b91c1c);
        }
        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.06);
        }
        .chip {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
        }
        .chip.success { background: #d4edda; color: #155724; }
        .chip.danger { background: #f8d7da; color: #721c24; }
        .chip.warning { background: #fff3cd; color: #856404; }
        .chip.info { background: #d1ecf1; color: #0c5460; }
        .chip.secondary { background: #e9ecef; color: #495057; }
        @media print {
            .no-print { display: none !important; }
            body { font-size: 12px; }
            .summary-card { 
                break-inside: avoid;
                margin-bottom: 10px;
                padding: 10px;
            }
            .summary-card .number { font-size: 1.5rem; }
            table { font-size: 10px; }
        }
        .semester-warning {
            background-color: #fef3c7;
            border-left: 4px solid #f59e0b;
            padding: 10px 15px;
            margin-bottom: 15px;
            border-radius: 6px;
        }
        .print-container {
            display: none;
        }
        .banner-img {
            max-width: 100%;
            height: auto;
            margin-bottom: 15px;
        }
        
        /* RTL Direction from right to left */
        body, .container, .card, .table, .form-control, .form-select {
            direction: rtl;
            text-align: right;
        }
        
        .ms-auto {
            margin-right: auto !important;
            margin-left: 0 !important;
        }
        
        .me-2 {
            margin-left: 0.5rem !important;
            margin-right: 0 !important;
        }
        
        .me-1 {
            margin-left: 0.25rem !important;
            margin-right: 0 !important;
        }
        
        .ms-2 {
            margin-right: 0.5rem !important;
            margin-left: 0 !important;
        }
        
        .ms-3 {
            margin-right: 1rem !important;
            margin-left: 0 !important;
        }
        
        .ms-auto {
            margin-right: auto !important;
            margin-left: 0 !important;
        }
        
        .text-end {
            text-align: left !important;
        }
        
        .text-right {
            text-align: right !important;
        }
        
        .border-end-0 {
            border-left: 0 !important;
            border-right: 1px solid #ced4da !important;
        }
        
        .border-start-0 {
            border-right: 0 !important;
            border-left: 1px solid #ced4da !important;
        }
        
        .input-group-text {
            border-radius: 0 8px 8px 0 !important;
        }
        
        .form-control {
            border-radius: 8px 0 0 8px !important;
        }
        
        .input-group {
            flex-direction: row-reverse;
        }
        
        .btn-group {
            flex-direction: row-reverse;
        }
        
        .d-flex {
            flex-direction: row-reverse;
        }
        
        .gap-2 > *:not(:last-child) {
            margin-left: 0.5rem;
            margin-right: 0;
        }
        
        .ps-3 {
            padding-right: 1rem !important;
            padding-left: 0 !important;
        }
        
        .py-4 {
            padding: 1.5rem !important;
        }
        
        .mb-4 {
            margin-bottom: 1.5rem !important;
        }
        
        .mb-3 {
            margin-bottom: 1rem !important;
        }
        
        .mb-2 {
            margin-bottom: 0.5rem !important;
        }
        
        .mb-1 {
            margin-bottom: 0.25rem !important;
        }
        
        .mt-3 {
            margin-top: 1rem !important;
        }
        
        .pe-3 {
            padding-left: 1rem !important;
            padding-right: 0 !important;
        }
        
        /* تصميم جديد لملخص التصفية */
        .filter-summary-card {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 1px solid #dee2e6;
            border-radius: 16px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .filter-summary-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #ef4444;
        }
        
        .filter-summary-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #ef4444, #dc2626);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.3rem;
            margin-left: 15px;
        }
        
        .filter-summary-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: #1f2937;
            margin: 0;
        }
        
        .filter-summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .filter-item {
            background: white;
            padding: 12px;
            border-radius: 10px;
            border: 1px solid #e5e7eb;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .filter-item-label {
            font-size: 0.85rem;
            color: #6b7280;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .filter-item-value {
            font-size: 1rem;
            font-weight: 600;
            color: #1f2937;
            word-break: break-word;
        }
        
        .filter-conditions {
            background: white;
            padding: 15px;
            border-radius: 10px;
            border: 1px solid #e5e7eb;
            margin-top: 15px;
        }
        
        .filter-conditions-title {
            font-size: 0.9rem;
            font-weight: 600;
            color: #374151;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .filter-conditions-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .filter-conditions-list li {
            padding: 5px 0;
            font-size: 0.85rem;
            color: #4b5563;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .filter-conditions-list li:before {
            content: "✓";
            color: #10b981;
            font-weight: bold;
        }
        
        .no-filters-message {
            text-align: center;
            padding: 30px 20px;
            color: #6b7280;
        }
        
        .no-filters-icon {
            font-size: 3rem;
            color: #d1d5db;
            margin-bottom: 15px;
        }
        
        /* تصميم البطاقات الإحصائية */
        .stats-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            border: 1px solid #e5e7eb;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            transition: transform 0.2s ease;
        }
        
        .stats-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 12px -1px rgba(0, 0, 0, 0.15);
        }
        
        .stats-card .number {
            font-size: 2.2rem;
            font-weight: 800;
            margin-bottom: 8px;
            line-height: 1;
        }
        
        .stats-card .label {
            font-size: 0.9rem;
            color: #6c757d;
            margin-bottom: 0;
        }
        
        .stats-card-icon {
            width: 60px;
            height: 60px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 15px;
            font-size: 1.5rem;
        }
        
        .stats-card-icon.red {
            background: linear-gradient(135deg, #fee2e2, #fecaca);
            color: #ef4444;
        }
        
        .stats-card-icon.orange {
            background: linear-gradient(135deg, #ffedd5, #fed7aa);
            color: #f97316;
        }
        
        .stats-card-icon.green {
            background: linear-gradient(135deg, #dcfce7, #bbf7d0);
            color: #22c55e;
        }
        
        .export-options {
            background: white;
            border-radius: 12px;
            padding: 20px;
            border: 1px solid #e5e7eb;
            margin-bottom: 20px;
        }
        
        .export-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .export-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-left: 15px;
            font-size: 1.1rem;
        }
        
        .export-title {
            font-size: 1rem;
            font-weight: 600;
            color: #1f2937;
            margin: 0;
        }
        
        .export-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .btn-export {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 10px 20px;
            border-radius: 10px;
            font-weight: 600;
            transition: all 0.3s ease;
            border: none;
        }
        
        .btn-export-excel {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
        }
        
        .btn-export-excel:hover {
            background: linear-gradient(135deg, #059669, #047857);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
        }
        
        .btn-export-pdf {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            color: white;
        }
        
        .btn-export-pdf:hover {
            background: linear-gradient(135deg, #dc2626, #b91c1c);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
        }
        
        @media (max-width: 768px) {
            .filter-summary-grid {
                grid-template-columns: 1fr;
            }
            
            .export-buttons {
                flex-direction: column;
            }
            
            .btn-export {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <!-- Hidden Print Container -->
    <div class="print-container" id="printContainer">
        <div class="print-header">
            <!-- Banner Image -->
            <img src="../img/banar3.jpg" alt="Banner" class="banner-img" onerror="this.style.display='none'">
            
            <h2>تقرير الطلاب غير المسجلين</h2>
            <div class="print-info">
                تاريخ التقرير: <?= date('Y-m-d H:i:s') ?><br>
                <?php if ($college_name): ?>الكلية: <?= e($college_name) ?><br><?php endif; ?>
                <?php if ($program_name): ?>البرنامج: <?= e($program_name) ?><br><?php endif; ?>
                <?php if ($semester_name): ?>الفصل الدراسي: <?= e($semester_name) ?><br><?php endif; ?>
            </div>
        </div>
        
        <table class="print-table">
            <thead>
                <tr>
                    <th width="5%">#</th>
                    <th width="15%">الرقم الجامعي</th>
                    <th width="50%">اسم الطالب</th>
                    <th width="15%">الدفعة</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($students)): ?>
                    <tr>
                        <td colspan="4" class="text-center">
                            جميع الطلاب مسجلين في الفصل الدراسي
                        </td>
                    </tr>
                <?php else: ?>
                    <?php 
                    $printTotalStudents = 0;
                    ?>
                    <?php foreach ($students as $index => $student): ?>
                        <?php
                        $printTotalStudents++;
                        ?>
                        <tr>
                            <td><?= $index + 1 ?></td>
                            <td><?= e($student['student_id']) ?></td>
                            <td class="text-right">
                                <?= e($student['student_name'] ?: $student['student_name_eng']) ?>
                            </td>
                            <td><?= e($student['batch_name']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                    
                    <!-- Total Row -->
                    <tr class="print-total">
                        <td colspan="3" class="text-end">
                            إجمالي الطلاب غير المسجلين: <?= number_format($printTotalStudents) ?>
                        </td>
                        <td></td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
        
        <div class="print-footer">
            تم إنشاء التقرير تلقائياً من نظام إدارة القبول والتسجيل<br>
            الصفحة 1 من 1
        </div>
    </div>
    
    <div class="container py-4">
        <!-- Header -->
        <div class="d-flex align-items-center mb-4 no-print" dir="ltr">
            <h3 class="mb-0 fw-bold">تقرير الطلاب غير المسجلين</h3>
            <div class="ms-auto">
                <span class="filter-status <?= $filters_applied ? 'applied' : 'not-applied' ?> me-2">
                    <i class="fas fa-<?= $filters_applied ? 'check-circle' : 'exclamation-triangle' ?> me-1"></i>
                    <?= $filters_applied ? 'تم تطبيق التصفية' : 'لم يتم تطبيق أي تصفية' ?>
                </span>
                <a class="btn btn-outline-secondary" href="../committee_dashboard.php">
                    <i class="fas fa-arrow-right me-2"></i>العودة للوحة التحكم
                </a>
            </div>
        </div>

        <?php if (!$filters_applied): ?>
            <div class="info-message no-print">
                <i class="fas fa-info-circle me-2"></i>
                <strong>تنبيه:</strong> لم يتم تطبيق أي تصفية. الرجاء اختيار عوامل التصفية والفصل الدراسي ثم النقر على "تطبيق التصفية" لعرض الطلاب غير المسجلين.
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="warning-message no-print">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <?= $error_message ?>
            </div>
        <?php endif; ?>

        <!-- Filter Card -->
        <div class="card card-arabic p-4 mb-4 no-print">
            <div class="row align-items-center">
                <div class="col-md-12">
                    <div class="d-flex align-items-center mb-4" dir="ltr">
                        <div class="filter-icon me-2" >
                            <i class="fas fa-filter"></i>
                        </div>
                        <h6 class="fw-bold mb-0 text-dark" >تصفية الطلاب غير المسجلين</h6>
                        <div class="ms-3">
                            <span class="badge bg-light text-dark border">
                                <i class="fas fa-sliders-h me-1"></i>
                                <?= $filters_applied ? 'تصفية مطبقة' : 'في انتظار التطبيق' ?>
                            </span>
                        </div>
                    </div>
                    
                    <form method="get" class="row g-3 align-items-end" id="filterForm">
                        <!-- Hidden field to indicate filters were applied -->
                        <input type="hidden" name="filter_applied" value="1">
                        
                        <!-- College Filter -->
                        <div class="col-lg-3 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-university me-1"></i>
                                    الكلية
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_college" id="filter_college" onchange="loadSections(this.value)">
                                        <option value="">اختر الكلية</option>
                                        <?php foreach ($colleges as $college): ?>
                                            <?php 
                                            // Don't show college filter if user has scope restriction
                                            if (!$canAll && $scopeCollege !== null && $college['id'] != $scopeCollege) {
                                                continue;
                                            }
                                            ?>
                                            <option value="<?= $college['id'] ?>" <?= ($filter_college_id == $college['id']) ? 'selected' : '' ?>>
                                                <?= e($college['name']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-building"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Section Filter (Dependent on College) -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-layer-group me-1"></i>
                                    القسم
                                </label>
                                <div class="input-group input-group-sm position-relative">
                                    <select class="form-control border-end-0" name="filter_section" id="filter_section" <?= !$filter_college_id ? 'disabled' : '' ?>>
                                        <option value="">جميع الأقسام</option>
                                        <?php if ($filter_college_id && !empty($sections)): ?>
                                            <?php foreach ($sections as $section): ?>
                                                <option value="<?= $section['id'] ?>" <?= ($filter_section_id == $section['id']) ? 'selected' : '' ?>>
                                                    <?= e($section['name']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-building"></i>
                                    </span>
                                </div>
                                <?php if (!$filter_college_id): ?>
                                    <small class="text-muted">الرجاء اختيار كلية أولاً</small>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Program Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-graduation-cap me-1"></i>
                                    البرنامج
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_program" id="filter_program">
                                        <option value="">جميع البرامج</option>
                                        <?php foreach ($programs as $program): ?>
                                            <option value="<?= $program['id'] ?>" <?= ($filter_program_id == $program['id']) ? 'selected' : '' ?>>
                                                <?= e($program['name']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-book"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Batch Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-users me-1"></i>
                                    الدفعة
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_batch">
                                        <option value="">جميع الدفعات</option>
                                        <?php foreach ($batches as $batch): ?>
                                            <option value="<?= $batch['id'] ?>" <?= ($filter_batch_id == $batch['id']) ? 'selected' : '' ?>>
                                                <?= e($batch['value']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-calendar-alt"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Semester Filter (Required) -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-calendar-alt me-1"></i>
                                    الفصل الدراسي <span class="text-danger">*</span>  <small class="text-danger">مطلوب</small>
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_semester" required>
                                        <option value="">اختر الفصل الدراسي</option>
                                        <?php foreach ($semesters as $semester): ?>
                                            <option value="<?= $semester['id'] ?>" <?= ($filter_semester_id == $semester['id']) ? 'selected' : '' ?>>
                                                <?= e($semester['name'] . ' (' . $semester['code'] . ')') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-calendar"></i>
                                    </span> 
                                </div>
                              
                            </div>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="col-12">
                            <div class="d-flex gap-2 pt-2">
                                <button type="submit" class="btn btn-primary btn-sm px-3">
                                    <i class="fas fa-filter me-1"></i>
                                    تطبيق التصفية
                                </button>
                                <a href="unregistered_students.php" class="btn btn-outline-secondary btn-sm px-3">
                                    <i class="fas fa-sync-alt me-1"></i>
                                    إعادة تعيين
                                </a>
                                <?php if ($filters_applied): ?>
                                    <span class="badge bg-danger align-self-center px-3 py-2">
                                        <i class="fas fa-check me-1"></i>
                                        تم تطبيق التصفية
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Report Content -->
        <?php if ($filters_applied && !empty($filter_semester_id)): ?>
            <!-- Semester Warning -->
            <div class="semester-warning no-print">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>ملاحظة:</strong> يتم عرض الطلاب غير المسجلين في الفصل الدراسي: 
                <span class="fw-bold"><?= e($semester_name) ?></span>
            </div>
            
            <!-- Filter Summary Card (Separate) -->
            <div class="filter-summary-card mb-4 no-print">
                <div class="filter-summary-header">
                    <div class="filter-summary-icon">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h5 class="filter-summary-title">ملخص التصفية المطبقة</h5>
                </div>
                
                <div class="filter-summary-grid">
                    <?php if ($filter_college_id && $college_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-university text-primary"></i>
                                الكلية
                            </div>
                            <div class="filter-item-value"><?= e($college_name) ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_section_id && $section_name && $filter_college_id && !empty($sections)): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-layer-group text-success"></i>
                                القسم
                            </div>
                            <div class="filter-item-value"><?= e($section_name) ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_program_id && $program_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-graduation-cap text-warning"></i>
                                البرنامج
                            </div>
                            <div class="filter-item-value"><?= e($program_name) ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_batch_id && $batch_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-users text-info"></i>
                                الدفعة
                            </div>
                            <div class="filter-item-value"><?= e($batch_name) ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_semester_id && $semester_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-calendar-alt text-danger"></i>
                                الفصل الدراسي
                            </div>
                            <div class="filter-item-value"><?= e($semester_name) ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <div class="filter-item">
                        <div class="filter-item-label">
                            <i class="fas fa-users text-secondary"></i>
                            عدد الطلاب
                        </div>
                        <div class="filter-item-value">
                            <span class="badge bg-danger fs-6"><?= number_format(count($students)) ?></span>
                        </div>
                    </div>
                </div>
                
      
            </div>
            
       

            <!-- Export Options -->
            <div class="export-options mb-4 no-print">
                <div class="export-header">
                    <div class="export-icon">
                        <i class="fas fa-download"></i>
                    </div>
                    <h5 class="export-title">خيارات التصدير والطباعة</h5>
                </div>
                
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <div class="text-muted mb-3 mb-md-0">
                            <i class="fas fa-info-circle me-2"></i>
                            <span>تاريخ التقرير: <?= date('Y-m-d H:i:s') ?></span>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="export-buttons justify-content-md-end">
                            <button class="btn btn-export btn-export-excel" onclick="exportToExcel()">
                                <i class="fas fa-file-excel"></i>
                                تصدير لإكسل
                            </button>
                            <button class="btn btn-export btn-export-pdf" onclick="printReportPDF()">
                                <i class="fas fa-print"></i>
                                طباعة PDF
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Students Table -->
            <div class="card card-arabic p-4">
                <div class="d-flex align-items-center mb-3" dir="ltr">
                    <div class="filter-icon me-2">
                        <i class="fas fa-users"></i>
                    </div>
                    <h6 class="fw-bold mb-0">
                        قائمة الطلاب غير المسجلين
                        <?php if ($program_name): ?>
                            <span class="chip warning ms-2">
                                <i class="fas fa-graduation-cap me-1"></i>
                                <?= e($program_name) ?>
                            </span>
                        <?php endif; ?>
                        <span class="chip danger ms-2">
                            <i class="fas fa-calendar-alt me-1"></i>
                            <?= e($semester_name) ?>
                        </span>
                    </h6>
                    <div class="ms-auto">
                        <span class="badge bg-danger fs-6">
                            <i class="fas fa-users me-1"></i>
                            <?= number_format(count($students)) ?> طالب
                        </span>
                    </div>
                </div>

                <div class="table-responsive">
                    <table class="table table-striped student-table" id="studentsTable">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>الرقم الجامعي</th>
                                <th>اسم الطالب</th>
                                <th>الكلية</th>
                                <th>القسم</th>
                                <th>البرنامج</th>
                                <th>الدفعة</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($students)): ?>
                                <tr>
                                    <td colspan="7" class="text-center text-success py-4">
                                        <div class="py-5">
                                            <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                                            <h5 class="text-success">جميع الطلاب مسجلين</h5>
                                            <p class="text-muted">جميع الطلاب مسجلين في الفصل الدراسي: <?= e($semester_name) ?></p>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($students as $index => $student): ?>
                                    <tr>
                                        <td class="fw-bold"><?= $index + 1 ?></td>
                                        <td><code><?= e($student['student_id']) ?></code></td>
                                        <td class="text-right" style="text-align: right; direction: rtl;">
                                            <?= e($student['student_name'] ?: $student['student_name_eng']) ?>
                                        </td>
                                        <td><?= e($student['college_name']) ?></td>
                                        <td><?= e($student['section_name']) ?></td>
                                        <td>
                                            <span class="chip info">
                                                <i class="fas fa-graduation-cap me-1"></i>
                                                <?= e($student['program_name']) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?= e($student['batch_name']) ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Simple section loading - reload page when college changes
        function loadSections(collegeId) {
            const sectionSelect = document.getElementById('filter_section');
            
            // Reset section select
            sectionSelect.innerHTML = '<option value="">جميع الأقسام</option>';
            
            if (!collegeId || collegeId === '') {
                sectionSelect.disabled = true;
                return;
            }
            
            // Create a form and submit it to reload the page
            const form = document.createElement('form');
            form.method = 'get';
            form.style.display = 'none';
            
            // Add college parameter
            const collegeInput = document.createElement('input');
            collegeInput.type = 'hidden';
            collegeInput.name = 'filter_college';
            collegeInput.value = collegeId;
            form.appendChild(collegeInput);
            
            // Add filter_applied parameter
            const appliedInput = document.createElement('input');
            appliedInput.type = 'hidden';
            appliedInput.name = 'filter_applied';
            appliedInput.value = '1';
            form.appendChild(appliedInput);
            
            // Add current section if exists
            const currentSection = "<?= $filter_section_id ?>";
            if (currentSection) {
                const sectionInput = document.createElement('input');
                sectionInput.type = 'hidden';
                sectionInput.name = 'filter_section';
                sectionInput.value = currentSection;
                form.appendChild(sectionInput);
            }
            
            // Add current program if exists
            const currentProgram = "<?= $filter_program_id ?>";
            if (currentProgram) {
                const programInput = document.createElement('input');
                programInput.type = 'hidden';
                programInput.name = 'filter_program';
                programInput.value = currentProgram;
                form.appendChild(programInput);
            }
            
            // Add current batch if exists
            const currentBatch = "<?= $filter_batch_id ?>";
            if (currentBatch) {
                const batchInput = document.createElement('input');
                batchInput.type = 'hidden';
                batchInput.name = 'filter_batch';
                batchInput.value = currentBatch;
                form.appendChild(batchInput);
            }
            
            // Add current semester if exists
            const currentSemester = "<?= $filter_semester_id ?>";
            if (currentSemester) {
                const semesterInput = document.createElement('input');
                semesterInput.type = 'hidden';
                semesterInput.name = 'filter_semester';
                semesterInput.value = currentSemester;
                form.appendChild(semesterInput);
            }
            
            document.body.appendChild(form);
            form.submit();
        }
        
        // Export to Excel function
        function exportToExcel() {
            // Get the table
            const table = document.getElementById('studentsTable');
            
            if (!table) {
                alert('لا توجد بيانات للتصدير');
                return;
            }
            
            // Create a new workbook
            const wb = XLSX.utils.book_new();
            
            // Create worksheet from table
            const ws = XLSX.utils.table_to_sheet(table);
            
            // Set column widths
            const wscols = [
                {wch: 5},   // #
                {wch: 15},  // Student ID
                {wch: 25},  // Student Name
                {wch: 15},  // College
                {wch: 15},  // Section
                {wch: 15},  // Program
                {wch: 10},  // Batch
            ];
            ws['!cols'] = wscols;
            
            // Add worksheet to workbook
            XLSX.utils.book_append_sheet(wb, ws, "الطلاب غير المسجلين");
            
            // Generate filename
            let filename = "الطلاب_غير_المسجلين";
            <?php if ($program_name): ?>
                filename += "_<?= e($program_name) ?>";
            <?php endif; ?>
            <?php if ($semester_name): ?>
                filename += "_<?= e($semester_name) ?>";
            <?php endif; ?>
            filename += "_<?= date('Y-m-d') ?>.xlsx";
            
            // Clean filename (remove special characters)
            filename = filename.replace(/[<>:"/\\|?*]+/g, '_');
            
            // Save the file
            XLSX.writeFile(wb, filename);
        }
        
        // PDF Print function with banner
        function printReportPDF() {
            if (!<?= $filters_applied ? 'true' : 'false' ?> || !<?= !empty($filter_semester_id) ? 'true' : 'false' ?>) {
                alert('الرجاء تطبيق التصفية أولاً مع اختيار فصل دراسي لعرض الطلاب');
                return;
            }
            
            if (<?= count($students) ?> === 0) {
                alert('لا توجد بيانات للطباعة');
                return;
            }
            
            // Create a new window for printing
            const printWindow = window.open('', '_blank', 'width=800,height=600');
            
            // Create the print content with banner
            const printContent = `
                <!DOCTYPE html>
                <html dir="rtl">
                <head>
                    <meta charset="UTF-8">
                    <title>تقرير الطلاب غير المسجلين</title>
                    <style>
                        @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap');
                        
                        * {
                            font-family: 'Cairo', sans-serif !important;
                            box-sizing: border-box;
                        }
                        
                        body {
                            margin: 0;
                            padding: 15mm;
                            direction: rtl;
                            background: white;
                            font-size: 12px;
                        }
                        
                        .print-header {
                            text-align: center;
                            margin-bottom: 20px;
                            padding-bottom: 10px;
                        }
                        
                        .banner-img {
                            max-width: 100%;
                            height: auto;
                            margin-bottom: 15px;
                        }
                        
                        .print-header h2 {
                            font-size: 20px;
                            margin: 5px 0;
                            font-weight: bold;
                            color: #000;
                        }
                        
                        .print-info {
                            font-size: 11px;
                            color: #666;
                            margin-top: 10px;
                            line-height: 1.6;
                        }
                        
                        .print-table {
                            width: 100%;
                            border-collapse: collapse;
                            font-size: 10px;
                            margin-top: 15px;
                        }
                        
                        .print-table th {
                            background-color: #f0f0f0;
                            border: 1px solid #000;
                            padding: 8px 5px;
                            font-weight: bold;
                            text-align: center;
                            font-size: 11px;
                        }
                        
                        .print-table td {
                            border: 1px solid #000;
                            padding: 6px 5px;
                            text-align: center;
                        }
                        
                        .print-table .text-right {
                            text-align: right;
                            padding-right: 10px;
                        }
                        
                        .print-total {
                            font-weight: bold;
                            background-color: #f8f9fa;
                            border-top: 2px solid #000;
                        }
                        
                        .print-total td {
                            padding: 8px 5px;
                        }
                        
                        .print-footer {
                            margin-top: 20px;
                            padding-top: 10px;
                            border-top: 1px solid #666;
                            font-size: 10px;
                            text-align: center;
                            color: #666;
                            line-height: 1.6;
                        }
                        
                        @media print {
                            @page {
                                size: A4 portrait;
                                margin: 15mm;
                            }
                            
                            body { 
                                margin: 0;
                                padding: 0;
                            }
                            
                            .print-table {
                                font-size: 9px;
                            }
                            
                            .print-table th, 
                            .print-table td {
                                padding: 5px 3px;
                            }
                        }
                    </style>
                </head>
                <body>
                    <div class="print-header">
                        <img src="../img/banar3.jpg" alt="Banner" class="banner-img" onerror="this.style.display='none'">
                        <h2>تقرير الطلاب غير المسجلين</h2>
                        <div class="print-info">
                            تاريخ التقرير: <?= date('Y-m-d H:i:s') ?><br>
                            <?php if ($college_name): ?>الكلية: <?= e($college_name) ?><br><?php endif; ?>
                            <?php if ($program_name): ?>البرنامج: <?= e($program_name) ?><br><?php endif; ?>
                            <?php if ($semester_name): ?>الفصل الدراسي: <?= e($semester_name) ?><br><?php endif; ?>
                        </div>
                    </div>
                    
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th width="5%">#</th>
                                <th width="15%">الرقم الجامعي</th>
                                <th width="50%">اسم الطالب</th>
                                <th width="15%">الدفعة</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($students)): ?>
                                <tr>
                                    <td colspan="4" class="text-center">
                                        جميع الطلاب مسجلين في الفصل الدراسي
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php 
                                $printTotalStudents = 0;
                                ?>
                                <?php foreach ($students as $index => $student): ?>
                                    <?php
                                    $printTotalStudents++;
                                    ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= e($student['student_id']) ?></td>
                                        <td class="text-right"><?= e($student['student_name'] ?: $student['student_name_eng']) ?></td>
                                        <td><?= e($student['batch_name']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                
                                <!-- Total Row -->
                                <tr class="print-total">
                                    <td colspan="3" class="text-end">
                                        إجمالي الطلاب غير المسجلين: <?= number_format($printTotalStudents) ?>
                                    </td>
                                    <td></td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    
                    <div class="print-footer">
                        تم إنشاء التقرير تلقائياً من نظام إدارة القبول والتسجيل<br>
                        الصفحة 1 من 1
                    </div>
                </body>
                </html>
            `;
            
            printWindow.document.write(printContent);
            printWindow.document.close();
            
            // Trigger print after content is loaded
            printWindow.onload = function() {
                printWindow.focus();
                setTimeout(function() {
                    printWindow.print();
                }, 500);
            };
        }
        
        // Initialize when page loads
        document.addEventListener('DOMContentLoaded', function() {
            const collegeSelect = document.getElementById('filter_college');
            const sectionSelect = document.getElementById('filter_section');
            
            // Enable/disable section select based on college selection
            if (collegeSelect && sectionSelect) {
                if (!collegeSelect.value) {
                    sectionSelect.disabled = true;
                } else {
                    sectionSelect.disabled = false;
                }
            }
        });
    </script>
    
    <!-- Load SheetJS library for Excel export -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
</body>
</html>