<?php

namespace App\Helpers;

use App\Models\Bank;
use App\Models\College;
use App\Models\Payment;
use App\Models\Registration;
use App\Models\Section;
use App\Models\Student;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

class UserHelper
{
    public static function getRequestFilters(): array
    {
        return request()->except('page');
    }

    public static function getCollegeId($user, array $filters): int
    {
        return $user->hasRole('admin')
            ? ($filters['college_id'] ?? 0)
            : $user->college_id;
    }

//    public static function getFilteredStudents($user, array $filters)
//    {
//        return UserHelper::userStudents($user, $filters, true, 30);
//    }

    public static function generateRandomStudentId(): string
    {
        return str_pad(rand(1, 99999999), 8, '0', STR_PAD_LEFT);
    }

    public static function userStudents($user, $filters, $pagination = true, $perPage = 10, $onlyRegistered = false)
    {
        $query = Student::with([
            'payments',
            'registrations',
            'registrations.semester',
            'payments.semester',
            'payments.bank',
            'card'
        ])->orderBy('student_name');

        if ($user->hasRole('registrar')) {
            $filters['college_id'] = $user->college_id;
        }

        // Apply search
        static::HandleSearch($query, $filters);

        // Apply filters only if they are not empty or "0"
        foreach ($filters as $column => $value) {
            if (!empty($value) && $value !== "0") {
                $query->where($column, $value);
            }
        }

        if ($onlyRegistered){
            $query->whereHas('payments', function ($q) {
                $q->where('payment_amount', '>', 0);
            });
        }

        return $pagination ? $query->paginate($perPage) : $query->get();
    }

    public static function userColleges($user)
    {
        return $user->hasRole('registrar')
            ? College::where('id', $user->college_id)->get()
            : College::all();
    }

    public static function userSections($collegeId)
    {
        return Section::where('parent', $collegeId)->get();
    }

    public static function userBanks($user)
    {
        // If the user has a financial role and a payment identifier, return specific banks
        if ($user->hasRole('financial') && $user->payment_identifier) {
            return Bank::where('bank_identifier', $user->payment_identifier)->get();
        }

        // Otherwise, return all banks
        return Bank::all();
    }

    public static function userPayments($user, $filters, $pagination = true, $perPage = 10)
    {
        $studentFilterFields = ['college_id', 'section_id', 'program_id', 'batch_id']; // define allowed student fields

        $query = Payment::with(['student', 'bank', 'semester', 'registration'])->orderBy('student_name');

        if ($user->hasRole('registrar')) {
            $filters['college_id'] = $user->college_id;
        }

        foreach ($filters as $column => $value) {
            if (empty($value) || $value === "0") {
                continue;
            }

            // Handle date range filters
            static::handleDateRangeFilters($query, $column, $value);

            // Handle constraint_id logic
            static::handleConstraintIdFilter($query, $column, $value);

            // Apply search
            static::HandleSearch($query, $filters);

            // Apply only valid student fields
            if (in_array($column, $studentFilterFields)) {
                $query->whereHas('student', function ($q) use ($column, $value) {
                    return $q->where($column, $value);
                });
            }
        }

        return $pagination ? $query->paginate($perPage) : $query->get();
    }

    public static function userFees($user, $filters, $pagination = true, $perPage = 10)
    {
        $studentFilterFields = ['college_id', 'section_id', 'program_id', 'batch_id']; // define allowed student fields

        $query = Registration::with(['student', 'semester'])->orderBy('student_name');

        if ($user->hasRole('registrar')) {
            $filters['college_id'] = $user->college_id;
        }

        foreach ($filters as $column => $value) {
            if (empty($value) || $value === "0") {
                continue;
            }

            // Handle date range filters
            static::handleDateRangeFilters($query, $column, $value);

            // Handle constraint_id logic
            static::handleConstraintIdFilter($query, $column, $value);

            // Apply search
            static::HandleSearch($query, $filters);

            // Apply only valid student fields
            if (in_array($column, $studentFilterFields)) {
                $query->whereHas('student', function ($q) use ($column, $value) {
                    return $q->where($column, $value);
                });
            }
        }

        return $pagination ? $query->paginate($perPage) : $query->get();
    }

    private static function handleDateRangeFilters(Builder $query, string $column, $value)
    {
        if (in_array($column, ['from', 'to'])) {
            $operator = $column === 'from' ? '>=' : '<=';
            try {
                $date = Carbon::createFromFormat('m/d/Y', $value)->format('Y-m-d');
                $query->where('created_at', $operator, $date);
            } catch (\Exception $e) {
                // Optional: log invalid date or skip
            }
        }
    }

    private static function handleConstraintIdFilter(Builder $query, string $column, $value)
    {
        if ($column === 'constraint_id') {
            $query->whereHas('registration', function ($q) use ($value) {
                if ($value == 1) {
                    // Fully registered
                    $q->whereRaw('payments.payment_amount = registrations.payment_amount');
                } elseif ($value == 2) {
                    // Partially registered
                    $q->whereRaw('payments.payment_amount != registrations.payment_amount')
                        ->whereRaw('payments.payment_amount >= (registrations.payment_amount / 2)');
                } elseif ($value == 3) {
                    // Unregistered
                    $q->whereRaw('payments.payment_amount < (registrations.payment_amount / 2)');
                }
            });
        }
    }

    private static function handleSearch(Builder $query, array &$filters): void
    {
        if (!empty($filters['keyword'])) {
            $keyword = $filters['keyword'];
            $keywordParts = array_filter(explode(' ', $keyword)); // remove empty strings

            $query->where(function ($q) use ($keywordParts) {
                $table = $q->getModel()->getTable();

                $hasStudentNumber = Schema::hasColumn($table, 'student_number');
                $hasStudentName = Schema::hasColumn($table, 'student_name');

                if ($hasStudentNumber || $hasStudentName) {
                    $q->where(function ($subQ) use ($hasStudentNumber, $hasStudentName, $keywordParts) {
                        foreach ($keywordParts as $part) {
                            $subQ->where(function ($innerQ) use ($hasStudentNumber, $hasStudentName, $part) {
                                if ($hasStudentNumber) {
                                    $innerQ->orWhere('student_number', 'like', "%{$part}%");
                                }
                                if ($hasStudentName) {
                                    $innerQ->orWhere('student_name', 'like', "%{$part}%");
                                }
                            });
                        }
                    });
                }
            });

            unset($filters['keyword']);
        }
    }
}
