<?php

namespace App\Http\Controllers\API;

use App\Http\Resources\StudentResource;
use LoggingHelper;
use App\Models\Bank;
use App\Models\Payment;
use App\Models\Student;
use App\Models\Semester;
use App\Models\Registration;
use Illuminate\Http\Request;
use App\Helpers\StudentHelper;
use App\Http\Controllers\Controller;
use App\Models\OtherFee;
use App\Models\OtherPayment;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use App\Helpers\SystemHelper;

class StudentPaymentController extends Controller
{
    public function addPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'payment_amount' => 'required',
            'payment_reference' => "required",
            'transaction_ref' => 'required',
            'semester_code' => 'required',
            'bank_identifier' => 'required',
            'bank_id' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $bankIdentifier = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if(!$bankIdentifier){
            $response = ['success' => "false", 'message' => "you are not authorized to perform this action"];

            $this->writeLog($request, $response);

            return response($response, 401);
        }

        $feeCode = substr($request->payment_reference, 0, 3);

        if($feeCode == 117) {
            return $this->otherFeePayment($request, $bankIdentifier);
        }

        return $this->studentFeePayment($request, $bankIdentifier);
    }

    public function otherFeePayment($request, $bankIdentifier)
    {
        $reference = $request->payment_reference;

        $fee = OtherFee::where(['reference' => $reference])->first();

        if(!$fee){
            $response = ['success' => "false", "message" => "fee not found"];

            $this->writeLog($request, $response);

            return response($response, 404);
        }

        if($request->payment_amount < $fee->amount + $fee->stamp){
            return response(['success' => false, 'message' => "payment amount is less than required"], 442);
        } else if($request->payment_amount > $fee->amount + $fee->stamp) {
            return response(['success' => false, 'message' => "payment amount is greater than required"], 442);
        }

        if($fee->is_paid == 1){

            $response = ['success' => "false", "message" => "this fee is paid"];

            $this->writeLog($request, $response);

            return response()->json($response, 442);
        }

        $otherPayment = new OtherPayment();

        $otherPayment['other_fee_id'] = $fee->id;
        $otherPayment['name'] = $fee->name;
        $otherPayment['phone_number'] = $fee->phone_number;
        $otherPayment['fee_type_id'] = $fee->fee_type_id;
        $otherPayment['reference'] = $request->payment_reference;
        $otherPayment['amount'] = $request->payment_amount;
        $otherPayment['payment_currency'] = $fee->payment_currency;
        $otherPayment['transaction_ref'] = $request->transaction_ref;
        $otherPayment['bank_id'] = $bankIdentifier->id;
        $otherPayment['stamp'] = $fee->stamp;

        $otherPayment->save();

        if ( !$otherPayment->save() && env('LOGGING_SYSTEM_ENABLED') == true ) {
            LoggingHelper::unsuccessfulPayment($request);
        }

        $fee->is_paid = 1;
        $fee->save();

        // $this->informMobileApp($payment['student_id'], $payment['semester_id'], $payment['payment_amount'], $payment['payment_reference'], $payment['created_at']);

        if (env('LOGGING_SYSTEM_ENABLED') == true) {
            LoggingHelper::successfulPayment($request);
        }

        return response()->json(['success' => true, 'data' => "", 'message' => "Payment has been added succssfully",], 200);

    }

    public function studentFeePayment($request, $bankIdentifier)
    {
        $student = Student::where('student_id', $request->student_id)->first();

        if(!$student){
            $response = ['success' => "false", "message" => "Student not found"];

            $this->writeLog($request, $response);

            return response($response, 404);
        }

        // chack if student is medically fit
        // $medicallyFit = $this->canPay($request->student_id);
        $medicallyFit = true;
        if( $medicallyFit != true ){

            $response = ['success' => "false", "message" => "Student is not medically fit"];

            $this->writeLog($request, $response);

            return response($response, 442);
        }

        $semester = Semester::where('code', $request->semester_code)->first();

        if(!$semester){

            $response = ['success' => "false", "message" => "semester not found, please check the semester code again"];

            $this->writeLog($request, $response);

            return response($response, 442);
        }

        $registration = Registration::where([ 'student_id' => $student->student_id, 'payment_reference' => $request->payment_reference, 'semester_id' => $semester->id ])->first();

        if(!$registration){

            $response = ['success' => "false", "message" => "payment not found"];

            $this->writeLog($request, $response);

            return response($response, 404);
        }

        $payment = new Payment();

        $penalty = $registration->penalty ?? 0.0;

        $feeAmount = $registration->payment_amount;

        if($request->payment_amount < $feeAmount){
            return response(['success' => false, 'message' => "payment amount ".$request->payment_amount." is less than required" . $feeAmount], 442);
        } else if($request->payment_amount > $feeAmount) {
            return response(['success' => false, 'message' => "payment amount is greater than required"], 442);
        }

        if($registration->is_paid == 1){

            $response = ['success' => "false", "message" => "Student fees are paid"];

            $this->writeLog($request, $response);

            return response()->json($response, 442);
        }

        $payment['student_id'] = $student->student_id;
        $payment['student_name'] = $student->student_name;

        $payment['payment_reference'] = $request->payment_reference;
        $payment['semester_id'] = Semester::where('code', $request->semester_code)->first()['id'];
        $payment['payment_amount'] = $request->payment_amount;
        $payment['payment_currency'] = $registration->payment_currency;
        $payment['transaction_ref'] = $request->transaction_ref;
        $payment['bank_id'] = $bankIdentifier->id;
        $payment['fee_type_id'] = $registration->fee_type_id;

        $payment['study_fees'] = $registration->study_fees;
        // $payment['study_fees'] = $registration->study_fees / 2;
        // $payment['study_fees'] = SystemHelper::studyFeeCorrection($payment, $registration);
        $payment['registration_fees'] = $registration->registration_fees;
        $payment['insurance_fees'] = $registration->insurance_fees;
        $payment['discount'] = $registration->discount;
        $payment['penalty'] = $penalty;
        $payment['stamp'] = $registration->stamp;

        $payment->save();

        if ( !$payment->save() && env('LOGGING_SYSTEM_ENABLED') == true ) {
            LoggingHelper::unsuccessfulPayment($request);
        }

        $registration->is_paid = 1;
        $registration->save();

        // $this->informMobileApp($payment['student_id'], $payment['semester_id'], $payment['payment_amount'], $payment['payment_reference'], $payment['created_at']);

        if (env('LOGGING_SYSTEM_ENABLED') == true) {
            LoggingHelper::successfulPayment($request);
        }

        return response()->json(['success' => true, 'data' => "", 'message' => "Payment has been added succssfully",], 200);

    }

    public function informMobileApp($studentNumber, $studentSemester, $paymentAmount, $paymentReference, $paymentCreatedAt)
    {
        $token = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJuYW1lIjoiUGF5bWVudHMiLCJyb2xlIjoiQWRtaW4iLCJuYmYiOjE2NzE5Nzk2NDMsImV4cCI6MTk4NzU5ODg0MywiaWF0IjoxNjcxOTc5NjQzLCJpc3MiOiJodHRwOi8vc21hcnR2YWxseS5jb20vIiwiYXVkIjoiaHR0cDovL3NtYXJ0dmFsbHkuY29tLyJ9.VZ0jlTkfvbqS0FBrRy-PaYrzlHNuJHxwAvnxMp2vZ3k";

        $response = Http::withToken($token)->post('http://3.19.233.152/api/payments/study-fees', [
            'studentNumber' => $studentNumber,
            'semester' => $studentSemester,
            'amount' => $paymentAmount,
            'referenceNumber' => $paymentReference,
            'paidAt' => $paymentCreatedAt,
        ]);

        return $response;
    }

    public function canPay($studentNumber)
    {
        $token = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJuYW1lIjoiUGF5bWVudHMiLCJyb2xlIjoiQWRtaW4iLCJuYmYiOjE2NzE5Nzk2NDMsImV4cCI6MTk4NzU5ODg0MywiaWF0IjoxNjcxOTc5NjQzLCJpc3MiOiJodHRwOi8vc21hcnR2YWxseS5jb20vIiwiYXVkIjoiaHR0cDovL3NtYXJ0dmFsbHkuY29tLyJ9.VZ0jlTkfvbqS0FBrRy-PaYrzlHNuJHxwAvnxMp2vZ3k";

        $response = Http::withToken($token)->get('http://3.19.233.152/api/Students/Medical', [
            'AdmissionFormNoOrStudentNumber' => $studentNumber
        ]);

        if (!empty($response['isMedicallyFit'])) {
            return $response['isMedicallyFit'];
        } else {
            return false;
        }

    }

    public function writeLog($request, $response)
    {
        if (env('LOGGING_SYSTEM_ENABLED') == true) {
            if($response["success"] == "false"){
                LoggingHelper::newPaymentRequestFail($request, $response);
            } else {

            }
        }
    }

    public function studentDetails(Request $request)
    {
        $bank = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if (!$bank) {
            return response()->json([
                'success' => false,
                'message' => 'You are not authorized to perform this action'
            ], 401);
        }

        $student = Student::with(['payments', 'registrations'])->where('student_id', $request->student_id)
            ->whereNull('deleted_at')
            ->first();

        if (!$student) {
            return response()->json([
                'success' => false,
                'code' => 404,
                'message' => 'No payments found for this student'
            ], 404);
        }

        return StudentResource::make($student);
    }
}
