<?php

namespace App\Http\Controllers\API;

use App\Helpers\LoggingHelper;
use App\Http\Controllers\Controller;
use App\Models\Bank;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;

class ThirdPartyController extends Controller
{
    private string $baseUrl;

    public function __construct()
    {
        $this->baseUrl = "https://api.eaeu.edu.sd/api";
    }

    public function studentInfo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'bank_identifier' => 'required'
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $bank = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if(!$bank){
            $response = ['success' => "false", 'message' => "you are not authorized to perform this action"];

            return response($response, 401);
        }

        return $this->getStudentData($request->student_id, $bank);
    }

    public function addPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'payment_amount' => 'required',
            'payment_reference' => "nullable",
            'transaction_ref' => 'required',
            'semester_code' => 'required',
            'bank_identifier' => 'required',
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $bank = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if(!$bank){
            $response = ['success' => "false", 'message' => "you are not authorized to perform this action"];

            return response($response, 401);
        }

        $student = $this->getStudentData($request->student_id, $bank);

        if ($student['success'] == true && $request->payment_amount != $student['data']['total']){
            return response()->json([
                "success" => false,
                "data" => "المبلغ غير مطابق",
                "message" => "يجب أن يطابق المبلغ المدفوع المبلغ المسجل للطالب"
            ]);
        }

        try {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $bank->access_token,
            ])->post("{$this->baseUrl}/pay", [
                'sno'       => $request->student_id,
                'phone'     => 1,
                'uuid'      => $request->transaction_ref,
                'total'     => $request->payment_amount,
                'bank_date' => now(),
                'currency'  => 1,
                'class_no'  => (int) filter_var($request->semester_code, FILTER_SANITIZE_NUMBER_INT),
            ]);

            return $response->json();

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function writeLog($request, $response)
    {
        if (env('LOGGING_SYSTEM_ENABLED') == true) {
            if($response["success"] == "false"){
                LoggingHelper::newPaymentRequestFail($request, $response);
            } else {

            }
        }
    }

    public function getStudentData($studentId, $bank)
    {
        try {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $bank->access_token,
            ])->post("{$this->baseUrl}/info", [
                'sno' => $studentId,
            ]);

            return $response->json();
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
