<?php

namespace App\Http\Controllers\API\V2;

use App\Helpers\LoggingHelper;
use App\Http\Controllers\Controller;
use App\Models\Bank;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;

class ThirdPartyController extends Controller
{
    private string $baseUrl;

    public function __construct()
    {
        $this->baseUrl = "https://api.eaeu.edu.sd/api";
    }

    public function studentInfo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'bank_identifier' => 'required'
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $bank = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if(!$bank){
            return response(['success' => false, 'message' => "you are not authorized to perform this action"], 401);
        }

        $response = $this->getStudentData($request->student_id, $bank);

        return $this->mapStudentResponse($response);
    }

    public function addPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'payment_amount' => 'required',
            'payment_reference' => "nullable",
            'transaction_ref' => 'required',
            'semester_code' => 'required',
            'bank_identifier' => 'required',
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $bank = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if(!$bank){
            return response(['success' => false, 'message' => "you are not authorized to perform this action"], 401);
        }

        $student = $this->getStudentData($request->student_id, $bank);

        if($student['success'] && $request->payment_amount < $student['data']['total']){
            return response(['success' => false, 'message' => "payment amount is less than required"], 442);
        } else if($student['success'] && $request->payment_amount > $student['data']['total']) {
            return response(['success' => false, 'message' => "payment amount is greater than required"], 442);
        }

        try {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $bank->access_token,

            ])->post("{$this->baseUrl}/pay", [
                'sno'       => $request->student_id,
                'phone'     => 1,
                'uuid'      => $request->transaction_ref,
                'total'     => $request->payment_amount,
                'bank_date' => now(),
                'currency'  => 1,
                'class_no'  => (int) filter_var($request->semester_code, FILTER_SANITIZE_NUMBER_INT),
            ]);

            if($response['success']){
                return response()->json(['success' => true, 'data' => "", 'message' => "Payment has been added successfully"], 200);
            } else if($response['message'] === "تم دفع الرسوم للطالب مسبقا") {
                return response()->json(['success' => "false", 'message' => "Student fees are paid"], 442);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function writeLog($request, $response)
    {
        if (env('LOGGING_SYSTEM_ENABLED') == true) {
            if($response["success"] == "false"){
                LoggingHelper::newPaymentRequestFail($request, $response);
            } else {

            }
        }
    }

    public function getStudentData($studentId, $bank)
    {
        $token = "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJhdWQiOiI5NjI4Y2Q0ZS01MzYyLTQxNGItYjUzNC0zOTA5NGU1NmZiYTgiLCJqdGkiOiJmZjg4YjVmMTU3NTVmYzI2NTVjMzgwMDE0NmU5ODhlNTRjMmUyNTBhM2U0OWNjMWZiM2VmMmYyMTA0YjQ5OTE5ZDhmODQ4MzczMTEyNjRmYiIsImlhdCI6MTc1ODEwODg0NC41NDI1MzIsIm5iZiI6MTc1ODEwODg0NC41NDI1MzQsImV4cCI6MTc4OTY0NDg0NC41MzIzMzUsInN1YiI6IjI3Iiwic2NvcGVzIjpbXX0.VvK6oB8bUaPOez19RBE0IewfiVHsTk8dqGPkIpdEbFAq1PuoAHDz70TlruZsVd4PLVFxUZF_z2KGSv07yn8dnL3HTwjvfFq7Idk1SYI2a7MfJBoVbpx4PGXrs1ceblD_NFXFbXc8ZDK6te5KwwVXC9SKaUhIB_jmrRL-o4O8jMfdad5uACuwapFuUDaFqNDL4BBAIlhtp-1syDo6J0xP_LTvHtyVVitzGYHzJt6rEri1a6O1IYUrRsUp0uyLAWO44CpaS4C2MitGOHtGKWb1caFf9pQ3LXqj9P1U7vmY89zdx_i1T6mQoe7VfFWAxI1fL-8SbKWk984ozQtnSzGe1qM41TWr48r2INJrX4Egp9IiBvazT5OBFaa9Mi5wzdKKenC90JCOmLhGytOYKdrpLW9zfYmp5y5R9k96oo3uoshnRV2POfvv1NUd6TO0JAXX6QY0fbtYRIqqPNv7Adrh8d-9B3AbQxmdE-9qVDRigwlD4ZbmWDx8GIX8eI0lh7r8INqzv723J5brPVJ_PwV5Wad1SQ128urjsXfTML5yQNgg5fxQFIPxjUdlv-FX8tY8kHWu-CRs8iTcaGYgwAJ-F7h49E5WafWT12vSIhlzfq0tzAdH1f_MbPdjDZ6QgOZjri1ZyQvhfSu6ICdShkx9eozdevbrW2hY-43N1irPFCA";
        try {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $bank->access_token,
            ])->post("{$this->baseUrl}/info", [
                'sno' => $studentId,
            ]);

            return $response->json();
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function mapStudentResponse($response)
    {
        if ($response instanceof \Illuminate\Http\JsonResponse) {
            $response = $response->getData(true); // convert JsonResponse to array
        }

        if ($response['success']){
            $data = $response['data'] ?? [];

            return response()->json([
                "success" => $response['success'] === "success", // convert to true/false
                "data" => [
                    "student_id"        => $data['student_no'] ?? null,     // map from student_no
                    "student_name"      => $data['student_name'] ?? null,
                    "student_college"   => $data['faculties'] ?? null,
                    "payment_amount"    => $data['total'] ?? null,
                    "payment_reference" => now()->format('YmdHis')  ,
                    "payment_currency"  => $this->mapCurrency($data['currency'] ?? null),
                    "semester_code"     => $this->mapSemester($data['class_no'] ?? null),
                ],
                "message" => $response['message'] ?? "",
            ], 200);

        } else if($response['data'] === "لا يوجد تسجيل مفتوح"){
            return response()->json(['success' => false, 'message' => "registration not set yet"], 442);

        } else if($response['data'] === "يوجد خطأ في الإدخال"){
            return response()->json(['success' => false, 'message' => "Student not found"], 404);
        }else {
            return response()->json(['success' => false, 'message' => "you are not authorized to perform this action"], 401);
        }
    }

    private function mapCurrency($currency): string
    {
        // You can adjust mapping as needed
        return match ($currency) {
            1       => "SDG",
            2       => "USD",
            default => "SDG",
        };
    }

    private function mapSemester($classNo): string
    {
        return match ($classNo) {
            1       => "1st",
            2       => "2nd",
            3       => "3rd",
            default => (string) $classNo,
        };
    }
}
