<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\UserHelper;
use App\Models\Batch;
use App\Models\FeeType;
use App\Models\Program;
use App\Models\Section;
use App\Models\Student;
use App\Models\Semester;
use App\Models\Registration;
use Illuminate\Http\Request;
use App\Helpers\StudentHelper;
use Illuminate\Support\Carbon;
use App\Http\Controllers\Controller;
use Brian2694\Toastr\Facades\Toastr;

class FeeController extends Controller
{
    /**
     * Provision a new web server.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|\Illuminate\Http\Response|\Illuminate\View\View
     */
    public function __invoke($studentId)
    {
        // Attempt to find the student by ID
        $student = Student::where('id', $studentId)->first();

        // If the student doesn't exist, show an error and redirect
        if (!$student) {
            Toastr::error('لم يتم ايجاد الطالب!', 'عملية غير ناجحة!', [
                "positionClass" => "toast-top-center"
            ]);

            return redirect()->back(); // use route name if available
        }

        // Load the view with necessary data
        return view('fees.create', [
            'student'   => $student,
            'feeTypes'  => FeeType::all(),
            'semesters' => Semester::all(),
        ]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\Response|\Illuminate\View\View
     */
    public function index()
    {

        $user = auth()->user();
        $filters = UserHelper::getRequestFilters();
        $collegeId = UserHelper::getCollegeId($user, $filters);
        $students = UserHelper::userStudents($user, $filters, true, 20);

        return view('fees.index', [
            'students'      => $students,
            'colleges'      => UserHelper::userColleges($user),
            'programs'      => Program::all(),
            'batches'       => Batch::all(),
            'sections'      => UserHelper::userSections($collegeId),
            'filters'       => (object) $filters,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        return "old create";
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $registration = new Registration();

        $registration->student_id = $request->student_id;
        $registration->student_name = $request->student_name;

        $registration->payment_reference = str_pad(rand(1, 999999999999), 12, "0", STR_PAD_LEFT);

        $registration->study_fees = $request->study_fees;
        $registration->registration_fees = $request->registration_fees;
        $registration->insurance_fees = $request->insurance_fees;
        $registration->discount = $request->discount;
        $registration->due_amount = $request->due_amount;
        $registration->payment_amount = $this->calculatePaymentAmount($request->registration_fees, $request->study_fees, $request->insurance_fees, $request->discount, $request->due_amount);
        $registration->payment_currency = $registration->payment_currency;

        $registration->semester_id = $request->semester_id;
        $registration->start = $request->registration_start;
        $registration->end = $request->registration_end;
        $registration->stamp = $request->stamp;
        $registration->user_id = auth()->user()->id;
        $registration->is_paid = 0;

        $registration->save();

        Toastr::success( __('fee.fee-add-message'), __('fee.success'), ["positionClass" => "toast-top-center"]);

        return redirect()->route('fee.show', $request->studentId);
    }

    public function show($id)
    {
        $student = Student::where('id', $id)->with(['payments', 'registrations'])->first();

        if(!$student){
            Toastr::error( 'لم يتم ايجاد الطالب!', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect('fee');
        }

        $studentFees = $student->registrations;

        foreach ($studentFees as $studentFee) {
            $payment = $student->payments->where('payment_reference', $studentFee->payment_reference)->first();
            $studentFee['paid_amount'] = $payment ? $payment->payment_amount : '0.0';
            $studentFee['semester'] = Semester::find($studentFee->semester_id);
        }

        return view('fees.show', [
            'studentFees' => $studentFees,
            'studentId' => $student->id,
            'studentName' => $student->student_name,
            'studentNotes' => $student->notes,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $type = explode("_", $id);
        $feeType = FeeType::where('code', $type[0])->first();

        $studentFee = Registration::where('payment_reference', $id)->first();

        if(!$studentFee){
            Toastr::error( 'لم يتم ايجاد الدفعية!', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect('fee');
        }

        $student = Student::find($studentFee->student_id);

        if(!$student){
            Toastr::error( 'لم يتم ايجاد الطالب!', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect('fee');
        }

        $studentFee['student_number'] = $student ? $student->student_number : '';
        $studentFee['national_id'] = $student ? $student->national_id : '';
        $studentFee['semester'] = Semester::find($studentFee->semester_id);
        $studentFee['fee_type_id'] = $feeType->id ?? null;
        $studentFee['payment_currency'] = $studentFee->payment_currency ?? 'SDG';

        return view('fees.edit', [
            'studentFee' => $studentFee,
            'student' => $student,
            'semesters' => Semester::all(),
            'feeTypes' => FeeType::all(),
        ]);

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $studentFee = Registration::where(['payment_reference' => $request->payment_reference])->first();

        if(!$studentFee){
            Toastr::error( 'لم يتم ايجاد الدفعية!', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect('fee');
        }

        $studentFee['student_name'] = $request->student_name;
        $studentFee['registration_fees'] = $request->registration_fees;
        $studentFee['study_fees'] = $request->study_fees;
        $studentFee['insurance_fees'] = $request->insurance_fees;
        $studentFee['discount'] = $request->discount;
        $studentFee['end'] = $request->registration_end ?? $studentFee->end;
        $studentFee['semester_id'] = $request->semester_id;
        $studentFee['due_amount'] = $request->due_amount;
        $studentFee['payment_amount'] = $request->fee_type_id > 1 ? $request->payment_amount : $this->calculatePaymentAmount($request->registration_fees, $request->study_fees, $request->insurance_fees, $request->discount, $request->due_amount);
        $studentFee['payment_currency'] = $request->payment_currency;
        $studentFee['stamp'] = $request->stamp ?? $studentFee->stamp;


        $studentFee->save();

        Toastr::success( __('fee.fee-edit-message'), __('fee.success'), ["positionClass" => "toast-top-center"]);
        return redirect()->route('fee.show', $request->std_id);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $studentFee = Registration::where(['payment_reference' => $id])->first();

        if (!$studentFee){
            return response()->json('error', 404);
        }

        $studentFee->delete();
        return response()->json('success', 200);
    }

    public function filterStudentFees(Request $request)
    {
        $students = UserHelper::userStudents(auth()->user());

        $sections = UserHelper::userSections($request->college_id);

        if ($request->has('filter')) {
            $students = StudentHelper::filterStudents($students, $request);
        }

        $colleges = UserHelper::userColleges(auth()->user());

        return view('fees.index', [
            'colleges' => $colleges,
//            'sections' => Section::where(['parent' => $request->filled('college_id') ? $request->college_id : 0])->get(),
            'sections' => $sections,
            'programs' => Program::all(),
            'batches' => Batch::all(),
            'semesters' => Semester::all(),
            'college_id' => $request->college_id,
            'section_id' => $request->section_id,
            'program_id' => $request->program_id,
            'batch_id' => $request->batch_id,
            'students' => $students,

        ]);
    }

    public function calculatePaymentAmount($registrationFees, $studyFees, $insuranceFees, $discount, $dueAmount)
    {
        if($registrationFees + $studyFees + $insuranceFees == 0){
            return 0;
        } else if ($studyFees == 0){
            return $registrationFees + $insuranceFees;
        } else {
            return ( ($studyFees + $registrationFees + $insuranceFees) - $discount ) + $dueAmount;
        }
    }

    public function getSemesterPaymentsNumber(Request $request)
    {
        $semester = Semester::find($request->semester_id);

        if (!$semester){
            return response()->json('error', 404);
        }

        $paymentsNumber = Registration::where(['student_id' => $request->student_id, 'semester_id' => $request->semester_id])->count();

        return response()->json($paymentsNumber, 200);
    }
}
