<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\UserHelper;
use App\Models\Batch;
use App\Models\Program;
use App\Models\Section;
use App\Models\Student;
use App\Models\Semester;
use App\Models\Registration;
use Illuminate\Http\Request;
use App\Helpers\StudentHelper;
use Illuminate\Support\Carbon;
use App\Http\Controllers\Controller;
use Brian2694\Toastr\Facades\Toastr;
use App\Models\RegistrationConstraint;
use Illuminate\Support\Facades\Validator;

class RegistrationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\Response|\Illuminate\View\View
     */
    public function index()
    {
//        $colleges = UserHelper::userColleges(auth()->user());
//
//        $sections = UserHelper::userSections(auth()->user()->college_id);
//
//        if(auth()->user()->hasRole('registrar')){
//            $students = UserHelper::userStudents(auth()->user());
//        } else {
//            $students = null;
//        }
//
//        return view('settings.setup-registration', [
//            'students' => $students,
//            'colleges' => $colleges,
//            'sections' => $sections,
//            'programs' => Program::all(),
//            'batches' => Batch::all()
//        ]);

        $user = auth()->user();

        // Get request filters except for pagination
        $filters = request()->except('page');

        // Determine the college ID based on the user's role
        $collegeId = $user->hasRole('admin')
            ? ($filters['college_id'] ?? 0)
            : $user->college_id;

        // Fetch students using helper with filters
        $students = UserHelper::userStudents($user, $filters);

        return view('settings.setup-registration', [
            'students'      => $students,
            'colleges'      => UserHelper::userColleges($user),
            'programs'      => Program::all(),
            'batches'       => Batch::all(),
            'semesters'     => Semester::all(),
            'sections'      => UserHelper::userSections($collegeId),
            'filters'       => (object) $filters,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'semester_id' => 'required|not_in:0',
        ]);

        if ($validator->fails()) {
            Toastr::error("يجب تحديد السمستر", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $data = collect($request->except('students'));

        if($this->durationInsideCalendar($data) == false){
            Toastr::error('يجب ان تكون بداية التسجيل و نهايته وفقا لتقويم العمادة !', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect()->route('settings.setup.registration');
        }

        if ($request->has('registration')) {
            $user = auth()->user();

            // Get request filters except for pagination
            $filters = request()->only(['college_id', 'section_id', 'program_id', 'batch_id']);

            // Determine the college ID based on the user's role
            $collegeId = $user->hasRole('admin')
                ? ($filters['college_id'] ?? 0)
                : $user->college_id;

            // Fetch students using helper with filters
            $students = UserHelper::userStudents($user, $filters, false);

            if($students){
                foreach ($students as $student) {
                    $registration = new Registration();
                    $registration->student_id = $student->student_id;
                    $registration->student_name = $student->student_name;
                    $registration->payment_reference = str_pad(rand(1, 999999999999), 12, "0", STR_PAD_LEFT);
                    $registration->study_fees = $student->program_fees;
                    $registration->registration_fees = $student->registration_fees;
                    $registration->insurance_fees = $student->insurance_fees;
                    $registration->discount = 0;
                    $registration->fee_type_id = 100;

                    $registration->is_paid = 0;
                    $registration->payment_amount = $this->calculatePaymentAmount($student->registration_fees, $student->program_fees, $student->insurance_fees, $request->stamp);
                    $registration->due_amount = 0;
                    $registration->semester_id = $request->semester_id;
                    $registration->start = $request->registration_start;
                    $registration->end = $request->registration_end;
                    $registration->stamp = $request->stamp;
                    $registration->user_id = auth()->user()->id;
                    $registration->save();
                }
            }

            Toastr::success('تم تفعيل التسجيل بنجاح!', 'عملية ناجحة!', ["positionClass" => "toast-top-center"]);

            return redirect()->route('settings.setup.registration');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $registration = Registration::find($id);

        if(!$registration){
            return response()->json('not found', 404);
        }

        $student = Student::find($registration->student_id);

        if(!$student){
            return response()->json('not found', 404);
        }

        $registration['college_id'] = $student['college_id'];
        $registration['section_id'] = $student['section_id'];
        $registration['batch_id'] = $student['batch_id'];
        $registration['program_id'] = $student['program_id'];

        return response()->json($registration, 200);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $registration = Registration::find($id);

        if(!$registration){
            return response()->json(false, 404);
        }

        $registrations = Registration::where('semester_id', $registration->semester_id)->get();

        foreach ($registrations as $registration) {
            $registration->delete();
        }

        return response()->json(true, 200);
    }

    public function manageRegistration()
    {
        if(auth()->user()->hasRole('registrar')){
            $registrations = Registration::with(['student', 'semester', 'student.college', 'student.batch'])
            ->where('user_id', auth()->user()->id)
            ->get()
            ->unique(function ($item) {
                return $item['semester_id'];
            });
        } else {
            $registrations = Registration::with(['student', 'semester', 'student.college', 'student.batch'])
                    ->get()
                    ->unique(function ($item) {
                        return $item['semester_id'];
                    });
        }

        if($registrations){
            foreach ($registrations as $registration) {
                if($registration->student){
                    $registration['college'] = $registration->student->college ? $registration->student->college->name : __('student.not-set');
                    $registration['batch'] = $registration->student->batch ? $registration->student->batch->value : __('student.not-set');
                } else {
                    $registration['college'] = __('student.not-set');
                }

                $registration['semester'] = $registration->semester ? $registration->semester->name : __('student.not-set');
            }
        }

        $colleges = UserHelper::userColleges(auth()->user());

        return view('settings.manage-registration',[
            'registrations' => $registrations,
            'colleges' => $colleges,
            'programs' => Program::all(),
            'batches' => Batch::all(),
        ]);
    }

//    public function manageRegistration()
//    {
//        $user = auth()->user();
//
//        // Subquery to get the latest (or earliest) registration per semester
//        $subQuery = Registration::selectRaw('MAX(id) as id')
//            ->when($user->hasRole('registrar'), function ($query) use ($user) {
//                $query->where('user_id', $user->id);
//            })
//            ->groupBy('semester_id');
//
//        // Main query using the subQuery result
//        $registrations = Registration::with([
//            'student',
//            'semester',
//            'student.college',
//            'student.batch'
//        ])
//            ->whereIn('id', $subQuery)
//            ->paginate(10);
//
//        // Transform results (adding college, batch, and semester display names)
//        $registrations->getCollection()->transform(function ($registration) {
//            $student = $registration->student;
//
//            $registration->college = $student?->college?->name ?? __('student.not-set');
//            $registration->batch = $student?->batch?->value ?? __('student.not-set');
//            $registration->semester = $registration->semester?->name ?? __('student.not-set');
//
//            return $registration;
//        });
//
//        return view('settings.manage-registration', [
//            'registrations' => $registrations,
//            'colleges'      => UserHelper::userColleges($user),
//            'programs'      => Program::all(),
//            'batches'       => Batch::all(),
//        ]);
//    }




    public function filterStudentsRegistration(Request $request)
    {
        $students = UserHelper::userStudents(auth()->user());

        $colleges = UserHelper::userColleges(auth()->user());

        $sections = UserHelper::userSections($request->college_id);

        if ($request->has('filter')) {
            $students = StudentHelper::filterStudents($students, $request);
        }

        return view('settings.setup-registration', [
            'colleges' => $colleges,
//            'sections' => Section::where(['parent' => $request->filled('college_id') ? $request->college_id : 0])->get(),
            'sections' => $sections,
            'programs' => Program::all(),
            'batches' => Batch::all(),
            'semesters' => Semester::all(),
            'college_id' => $request->college_id,
            'section_id' => $request->section_id,
            'program_id' => $request->program_id,
            'payment_currency' => $request->payment_currency,
            'batch_id' => $request->batch_id,
            'students' => $students,

        ]);
    }

    public function showRegistrationCollection($id)
    {
        $registration = Registration::find($id);

        if(!$registration){
            return response()->json('not found', 404);
        }

        $registrations = Registration::with('semester', 'student', 'student.college', 'student.section', 'student.program', 'student.batch')
            ->where('semester_id', $registration->semester_id)
            ->get();

        if(!$registrations){
            return response()->json('not found', 404);
        }

        foreach ($registrations as $registration) {
            $registration['college_name'] = $registration->student->college->name ?? 'لم يتم التحديد';
            $registration['section_name'] = $registration->student->section->name ?? 'لم يتم التحديد';
            $registration['program_name'] = $registration->student->program->name ?? 'لم يتم التحديد';
            $registration['batch_name'] = $registration->student->batch->value ?? 'لم يتم التحديد';
        }

        return response()->json($registrations, 200);
    }
    public function editRegistrationCollection($id)
    {
        $registration = Registration::find($id);

        if(!$registration){
            return response()->json('not found', 404);
        }

        $student = Student::find($registration->student_id);

        if(!$student){
            return response()->json('not found', 404);
        }

        $registration['college_id'] = $student['college_id'];
        $registration['section_id'] = $student['section_id'];
        $registration['batch_id'] = $student['batch_id'];
        $registration['program_id'] = $student['program_id'];

        $registration['start'] = date('Y-m-d', strtotime($registration['start']));
        $registration['end'] = date('Y-m-d', strtotime($registration['end']));
        $registration['stamp'] = $registration->stamp > 0.00 ? $registration->stamp : "50.00";

        $registration['sections'] = Section::where('parent', $student['college_id'])->get();

        return response()->json($registration, 200);
    }

    public function updateRegistrationCollection(Request $request)
    {
        $data = collect($request->except('students'));

        if (!$this->durationInsideCalendar($data)) {
            Toastr::error('يجب ان تكون بداية التسجيل و نهايته وفقا لتقويم العمادة !', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $studentsIds = Student::where('college_id', $request->college_id)
            ->where('batch_id', $request->batch_id)
            ->pluck('student_id');

        if ($studentsIds->isEmpty()) {
            Toastr::error('لم يتم ايجاد المجموعة!', 'عملية غير ناجحة!', ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        // Perform batch update instead of looping through each student
        Registration::whereIn('student_id', $studentsIds)->update([
            'start' => $request->registration_start,
            'end' => $request->registration_end,
            'stamp' => $request->stamp,
        ]);

        Toastr::success('تم تحديث المجموعة بنجاح!', 'عملية ناجحة!', ["positionClass" => "toast-top-center"]);
        return redirect()->back();
    }

    public function calculatePaymentAmount($registrationFees, $studyFees, $insuranceFees, $stamp)
    {
        if( ($registrationFees + $studyFees + $insuranceFees) == 0){
            return 0;
        } else if ($studyFees == 0){
            return $registrationFees + $insuranceFees;
        } else {
            return $studyFees + $registrationFees + $insuranceFees + $stamp;
        }
    }

    public function durationInsideCalendar($data)
    {
        return true;
        $registration = RegistrationConstraint::where(['college_id' => $data['college_id'], 'batch_id' => $data['batch_id']])->first();

        if(!$registration){
            return false;
        }

        $start = Carbon::parse($registration->start);
        $end = Carbon::parse($registration->end);

        if(Carbon::parse($data['registration_start'])->between($start, $end) && Carbon::parse($data['registration_end'])->between($start, $end)){
            return true;
        } else {
            return false;
        }
    }
}
