<?php

namespace App\Http\Controllers\Dashboard;

use App\Models\Bank;
use App\Models\Batch;
use App\Models\College;
use App\Models\Payment;
use App\Models\Program;
use App\Models\Section;
use App\Models\Student;
use App\Models\Semester;
use App\Models\Admission;
use App\Models\Registration;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Http\Controllers\Controller;
use App\Helpers\StudentHelper;
use App\Helpers\UserHelper;
use Illuminate\Support\Facades\DB;
class ReportController extends Controller
{
    public function index()
    {
        return view('reports.index');
    }

    public function payments()
    {
        $user = auth()->user();
        $filters = UserHelper::getRequestFilters();
        $collegeId = UserHelper::getCollegeId($user, $filters);
        $studentsPayments = UserHelper::userPayments($user, $filters, true, 20);

        return view('reports.payments', [
            'studentsPayments'  => $studentsPayments,
            'admissions'        => Admission::all(),
            'colleges'          => UserHelper::userColleges($user),
            'sections'          => UserHelper::userSections($collegeId),
            'programs'          => Program::all(),
            'batches'           => Batch::all(),
            'banks'             => Bank::all(),
            'filters'           => (object) $filters,
        ]);
    }

    public function fees()
    {
        $user = auth()->user();
        $filters = UserHelper::getRequestFilters();
        $collegeId = UserHelper::getCollegeId($user, $filters);
        $studentsFees = UserHelper::userFees($user, $filters, true, 20);

        return view('reports.fees', [
            'studentsFees'      => $studentsFees,
            'admissions'        => Admission::all(),
            'colleges'          => UserHelper::userColleges($user),
            'sections'          => UserHelper::userSections($collegeId),
            'programs'          => Program::all(),
            'batches'           => Batch::all(),
            'banks'             => Bank::all(),
            'filters'           => (object) $filters,
        ]);
    }

    public function revenues(Request $request)
    {
        $user = auth()->user();

        $banks = UserHelper::userBanks($user);
        $colleges = UserHelper::userColleges($user);

        // --- Base Query ---
        $paymentsQuery = Payment::query()
            ->join('registrations', 'payments.payment_reference', '=', 'registrations.payment_reference')
            ->join('students', 'payments.student_id', '=', 'students.student_id')
            ->select([
                'payments.student_id',
                'payments.student_name',
                'payments.payment_reference',
                'payments.semester_id',
                'payments.transaction_ref',
                'payments.bank_id',
                'payments.deleted_at',

                'payments.payment_amount as p_payment_amount',
                'payments.study_fees as p_study_fees',
                'payments.registration_fees as p_registration_fees',
                'payments.insurance_fees as p_insurance_fees',
                'payments.discount as p_discount',

                'registrations.payment_amount as r_payment_amount',

                'students.college_id',
                'students.section_id',
                'students.program_id',
                'students.batch_id',

                'payments.created_at as p_created_at',
                'payments.updated_at as p_updated_at',
            ]);

        // --- Apply Role-Based Filtering ---
        if ($user->hasRole('registrar')) {
            $paymentsQuery->where('students.college_id', $user->college_id);
        } elseif ($user->hasRole('admin')) {
            if ($request->college_id) {
                $paymentsQuery->where('students.college_id', $request->college_id);
            }
        } else {
            $payments = collect(); // empty collection
            return $this->returnView($payments, $colleges, $banks, $request);
        }

        // --- Fetch Results ---
        $payments = $paymentsQuery->get();

        // --- Add constraint_id ---
        $payments->transform(function ($payment) {
            $r = $payment->r_payment_amount;
            $p = $payment->p_payment_amount;

            if ($r == $p && $r != 0) {
                $payment->constraint_id = 1;
            } elseif ($r != $p && $p > ($r / 2)) {
                $payment->constraint_id = 2;
            } else {
                $payment->constraint_id = 3;
            }

            return $payment;
        });

        return $this->returnView($payments, $colleges, $banks, $request);
    }

    /**
     * Extracted to avoid duplication
     */
    private function returnView($payments, $colleges, $banks, $request)
    {
        return view('reports.revenues', [
            'payments'   => $payments,
            'colleges'   => $colleges,
            'programs'   => Program::all(),
            'batches'    => Batch::all(),
            'admissions' => Admission::all(),
            'banks'      => $banks,
            'filters'    => $request,
        ]);
    }


    public function revenuesMin()
    {
//        return request()->all();
        // Get all banks
        $banks = Bank::all();

        // Get aggregated payment data by college and admission_id
        $paymentSums = Payment::with('student.college')
            ->select(
                'students.college_id',
                'students.admission_id',
                DB::raw('SUM(payments.study_fees) as total_study_fees'),
                DB::raw('SUM(payments.registration_fees) as total_registration_fees'),
                DB::raw('SUM(payments.insurance_fees) as total_insurance_fees'),
                DB::raw('SUM(payments.discount) as total_discount'),
                DB::raw('SUM(payments.penalty) as total_penalty'),
                DB::raw('SUM(payments.stamp) as total_stamp')
            )
            ->join('students', 'students.student_id', '=', 'payments.student_id')
            ->groupBy('students.college_id', 'students.admission_id')
            ->get();

        // Prepare a structure to accumulate data by college
        $collegeData = [];

        foreach ($paymentSums as $row) {
            $collegeId = $row->college_id;
            $admissionId = $row->admission_id;

            // Ensure default values exist
            if (!isset($collegeData[$collegeId])) {
                $collegeData[$collegeId] = [
                    'total_private_fees' => 0,
                    'total_not_private_fees' => 0,
                    'total_other_fees' => 0,
                    'total_registration_fees' => 0,
                    'total_insurance_fees' => 0,
                    'total_discount' => 0,
                    'total_penalty' => 0,
                    'total_stamp' => 0,
                ];
            }

            // Categorize study_fees by admission type
            if ($admissionId == 2) {
                $collegeData[$collegeId]['total_private_fees'] += $row->total_study_fees;
            } elseif ($admissionId == 101) {
                $collegeData[$collegeId]['total_other_fees'] += $row->total_study_fees;
            } else {
                $collegeData[$collegeId]['total_not_private_fees'] += $row->total_study_fees;
            }

            // Accumulate other fees
            $collegeData[$collegeId]['total_registration_fees'] += $row->total_registration_fees;
            $collegeData[$collegeId]['total_insurance_fees'] += $row->total_insurance_fees;
            $collegeData[$collegeId]['total_discount'] += $row->total_discount;
            $collegeData[$collegeId]['total_penalty'] += $row->total_penalty;
            $collegeData[$collegeId]['total_stamp'] += $row->total_stamp;
        }

        // Load all colleges and attach the aggregated data as dynamic attributes
        $colleges = College::all()->map(function ($college) use ($collegeData) {
            $data = $collegeData[$college->id] ?? [
                'total_private_fees' => 0,
                'total_not_private_fees' => 0,
                'total_other_fees' => 0,
                'total_registration_fees' => 0,
                'total_insurance_fees' => 0,
                'total_discount' => 0,
                'total_penalty' => 0,
                'total_stamp' => 0,
            ];

            foreach ($data as $key => $value) {
                $college->$key = $value; // Dynamically attach the attribute to the College model
            }

            return $college;
        });

        return view('reports.revenue-min', [
            'colleges' => $colleges,
            'banks' => $banks,
        ]);
    }

    public function showStudentFees($studentId)
    {
        dd($studentId);
        $student = Student::where('student_id', $studentId)
            ->with(['registrations', 'payments'])
            ->first();

        if($student){
            $studentFees = Registration::where('student_id', $student->student_id)->get();

            foreach ($studentFees as $studentFee) {
                $payment = Payment::where('payment_reference', $studentFee->payment_reference)->first();

                $studentFee['paid_amount'] = $payment ? $payment['payment_amount'] : '0.0';
            }
        }

        return view('reports.show-student-fees', [
            'studentFees' => isset($studentFees) ? $studentFees : null,
            'student' => $student,
        ]);

    }

    public function showStudentPayments($studentId)
    {
        $student = Student::where('id', $studentId)->first();

        $studentPayments = $student
            ? Payment::where('student_id', $student->student_id)->get()
            : collect();

        return view('reports.show-student-payments', [
            'studentPayments' => $studentPayments,
            'totalPaid' => $studentPayments->sum('payment_amount'),
            'student_name' => $student->student_name,
        ]);
    }
}
