<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\FileHelper;
use App\Helpers\StudentHelper;
use App\Helpers\UserHelper;
use App\Models\Bank;
use App\Models\File;
use App\Models\Batch;
use App\Models\State;
use App\Models\Payment;
use App\Models\Program;
use App\Models\Section;
use App\Models\Student;
use App\Models\FileType;
use App\Models\Semester;
use App\Models\Admission;
use App\Models\Scholarship;
use App\Models\AcademicYear;
use App\Models\Registration;
use Illuminate\Http\Request;
use App\Models\EducationStatus;
use App\Models\ScholarshipDonor;
use App\Models\ScholarshipFunding;
use App\Models\StudentScholarShip;
use App\Http\Controllers\Controller;
use App\Models\Locality;
use Brian2694\Toastr\Facades\Toastr;
use App\Models\PrimaryCertificateType;
use App\Models\PrimaryCertificateYear;
use App\Models\StudentEducationStatus;
use App\Models\StudentPrimaryCertificate;
use Illuminate\Support\Facades\Validator;

class StudentController extends Controller
{

//    public function index()
//    {
//        $user = auth()->user();
//        $filters = UserHelper::getRequestFilters();
//        $collegeId = UserHelper::getCollegeId($user, $filters);
//        $students = UserHelper::userStudents($user, $filters, true, 20);
//
//        return view('students.index', [
//            'students'      => $students,
//            'colleges'      => UserHelper::userColleges($user),
//            'programs'      => Program::all(),
//            'batches'       => Batch::all(),
//            'sections'      => UserHelper::userSections($collegeId),
//            'newStudentId'  => str_pad(rand(1, 99999999), 8, '0', STR_PAD_LEFT),
//            'filters'       => (object) $filters,
//        ]);
//    }


//    public function index(Request $request)
//    {
//        $user = auth()->user();
//        $filters = UserHelper::getRequestFilters();
//        $collegeId = UserHelper::getCollegeId($user, $filters);
//        $keyword = $request->input('keyword');
//
//        if ($request->ajax() && !empty($keyword)) {
//            // Handle AJAX search
//            $students = Student::query()
//                ->where(function ($query) use ($keyword) {
//                    $query->where('student_number', 'like', "%{$keyword}%")
//                        ->orWhere('student_name', 'like', "%{$keyword}%");
//                })
//                ->orderBy('id', 'desc')
//                ->paginate(10);
//
//            return response()->json([
//                'html' => view('students.partials.table', compact('students'))->render()
//            ]);
//        } else {
//            // Standard page load
//            $students = UserHelper::userStudents($user, $filters, true, 20);
//
//            return view('students.index', [
//                'students'      => $students,
//                'colleges'      => UserHelper::userColleges($user),
//                'programs'      => Program::all(),
//                'batches'       => Batch::all(),
//                'sections'      => UserHelper::userSections($collegeId),
//                'newStudentId'  => str_pad(rand(1, 99999999), 8, '0', STR_PAD_LEFT),
//                'filters'       => (object) $filters,
//            ]);
//        }
//    }

    public function index(Request $request)
    {
        $user = auth()->user();
        $filters = UserHelper::getRequestFilters();
        $collegeId = UserHelper::getCollegeId($user, $filters);
        $students = UserHelper::userStudents($user, $filters, true, 20);

        $data = [
            'students'      => $students,
            'colleges'      => UserHelper::userColleges($user),
            'programs'      => Program::all(),
            'batches'       => Batch::all(),
            'sections'      => UserHelper::userSections($collegeId),
            'newStudentId'  => UserHelper::generateRandomStudentId(),
            'filters'       => (object) $filters,
        ];

        if ($request->ajax()) {
            return response()->json([
                'html' => view('students.partials.table', ['students' => $students])->render()
            ]);
        }

        return view('students.index', $data);
    }



    public function create()
    {
        $colleges = UserHelper::userColleges(auth()->user());

        return view('students.create',[
            'colleges' => $colleges,
            'programs' => Program::all(),
            'batches' => Batch::all(),
            'states' => State::all(),
            'admissions' => Admission::all(),
            'newStudentId' => str_pad(rand(1, 99999999), 8, "0", STR_PAD_LEFT),
            'primaryCertificateTypes' => PrimaryCertificateType::all(),
            'primaryCertificateYears' => PrimaryCertificateYear::all(),
            'scholarships' => Scholarship::all(),
            'scholarshipFundings' => ScholarshipFunding::all(),
            'scholarshipDonors' => ScholarshipDonor::all(),
            'educationStatuses' => EducationStatus::all(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\RedirectResponse|\Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'student_name' => "required",
            'admission_id' => 'required',
            'national_id' => 'nullable',
            'program_id' => 'required',
            'program_fees' => 'required',
            'registration_fees' => 'required',
            'section_id' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $student = Student::where('student_id', $request->student_id)->first();

        if ($student) {
            Toastr::error("تم استخدام الرقم الجامعي  (" .$request->student_id. ") واسناده الي الطالب ".$student->student_name, __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->route('student.index');
        }

        $student = new Student();
        $student['student_id'] = $request->student_id;
        $student['student_number'] = $request->student_id;
        $student['student_name'] = $request->student_name;
        $student['gender'] = $request->gender;
        $student['admission_id'] = $request->admission_id;
        $student['national_id'] = $request->national_id;

        $student['college_id'] = $request->college_id;
        $student['section_id'] = $request->section_id;
        $student['program_id'] = $request->program_id;
        $student['batch_id'] = $request->batch_id;

        $student['program_fees'] = $request->program_fees;
        $student['registration_fees'] = $request->registration_fees;
        $student['insurance_fees'] = $request->insurance_fees;
        $student['phone_number'] = $request->phone_number;
        $student['phone_number2'] = $request->phone_number2;
        $student['date_of_birth'] = $request->date_of_birth;
        $student['notes'] = $request->notes;

        $student['accommodation_type'] = $request->accommodation_type;
        $student['acceptance_rate'] = $request->acceptance_rate;
        $student['state_id'] = $request->state_id;
        $student['locality_id'] = $request->locality_id;
        $student['guardian_name'] = $request->guardian_name;
        $student['guardian_address'] = $request->guardian_address;
        $student['guardian_phone_number'] = $request->guardian_phone_number;



        $student->save();

        if ($request->primary_certificate_type && $request->primary_certificate_type) {
            $certificate = new StudentPrimaryCertificate();
            $certificate->student_id = $student->student_id;
            $certificate->certificate_type_id = $request->primary_certificate_type;
            $certificate->certificate_year_id = $request->primary_certificate_type;
            $certificate->save();
        }

        if ($request->scholarship && $request->scholarship_funding && $request->scholarship_donor){
            $scholarship = new StudentScholarShip();
            $scholarship->student_id = $student->student_id;
            $scholarship->scholarship_id = $request->scholarship;
            $scholarship->scholarship_funding_id = $request->scholarship_funding;
            $scholarship->scholarship_donor_id = $request->scholarship_donor;
            $scholarship->save();
        }

        if($request->education_status){
            $status = new StudentEducationStatus();
            $status->student_id = $student->student_id;
            $status->education_status_id = $request->education_status;
            $status->save();
        }

        if($request->has('student_image')){
            $uploadedFilePath = FileHelper::uploadFile($request->file('student_image'), $request->student_id);

            $file = new File();

            $file->file_type_id = 1;
            $file->fileable_id = $request->student_id;
            $file->fileable_type = "App\Models\Student";
            $file->note = "تم الاضافة بواسطة مدير";
            $file->url = $uploadedFilePath;

            $file->save();
        }

        Toastr::success( __('student.student-add-message'), __('student.success'), ["positionClass" => "toast-top-center"]);
        return redirect()->route('student.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){}

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function edit($id)
    {
        $student = Student::where(['id' => $id])->first();

        if(!$student){
            return redirect()->back()->withErrors('not found')->withInput();
        }

        $student['image'] = $student->files->first()->url ?? 'no-image.png';

        $colleges = UserHelper::userColleges(auth()->user());

        return view('students.edit', [
            'student' => $student,
            'years' => AcademicYear::all(),
            'states' => State::all(),

            'colleges' => $colleges,
            'sections' => Section::where('parent', $student->college_id)->get(),
            'programs' => Program::all(),
            'batches' => Batch::all(),

            'admissions' => Admission::all(),
            'newStudentId' => str_pad(rand(1, 99999999), 8, "0", STR_PAD_LEFT),
            'primaryCertificateTypes' => PrimaryCertificateType::all(),
            'primaryCertificateYears' => PrimaryCertificateYear::all(),
            'scholarships' => Scholarship::all(),
            'scholarshipFundings' => ScholarshipFunding::all(),
            'scholarshipDonors' => ScholarshipDonor::all(),
            'educationStatuses' => EducationStatus::all(),
            'semesters' => Semester::all(),

            'oldScholarship' => StudentScholarShip::where('student_id', $student->student_id)->first(),
            'oldPrimaryCertificate' => StudentPrimaryCertificate::where('student_id', $student->student_id)->first(),
            'oldStatus' => StudentEducationStatus::where('student_id', $student->student_id)->first(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\RedirectResponse|\Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'student_name' => "required",
            'admission_id' => 'required',
            'national_id' => 'nullable',
            'program_id' => 'required',
            'program_fees' => 'required',
            'registration_fees' => 'required',
            'section_id' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response(['errors' => $validator->errors()->all()], 422);
        }

        $student = Student::where(['id' => $id])->first();

        if (!$student) {
            return response(['success' => false, 'message' => "Student not found",], 404);
        }

        $student['student_id'] = $request->student_id;
        $student['student_number'] = $request->student_id;
        $student['student_name'] = $request->student_name;
        $student['gender'] = $request->gender;
        $student['admission_id'] = $request->admission_id;
        $student['national_id'] = $request->national_id;

        $student['college_id'] = $request->college_id;
        $student['section_id'] = $request->section_id ?? $student->section_id;
        $student['program_id'] = $request->program_id;
        $student['batch_id'] = $request->batch_id;

        $student['program_fees'] = $request->program_fees;
        $student['registration_fees'] = $request->registration_fees;
        $student['insurance_fees'] = $request->insurance_fees;
        $student['phone_number'] = $request->phone_number;
        $student['phone_number2'] = $request->phone_number2;
//        $student['guardian_phone_number'] = $request->guardian_phone_number;
        $student['date_of_birth'] = $request->date_of_birth;
        $student['notes'] = $request->notes;

        $student['accommodation_type'] = $request->accommodation_type;
        $student['acceptance_rate'] = $request->acceptance_rate;
        $student['state_id'] = $request->state_id;
        $student['locality_id'] = $request->locality_id;
        $student['guardian_name'] = $request->guardian_name;
        $student['guardian_address'] = $request->guardian_address;
        $student['guardian_phone_number'] = $request->guardian_phone_number;

        $student->save();

        if($request->has('student_image')){

            FileHelper::deleteOldFile('/files/students/', $request->oldImage, $student);

            $uploadedFilePath = FileHelper::uploadFile($request->file('student_image'), $request->student_id);

            $file = new File();

            $file->file_type_id = 1;
            $file->fileable_id = $request->student_id;
            $file->fileable_type = "App\Models\Student";
            $file->note = "تم الاضافة بواسطة مدير";
            $file->url = $uploadedFilePath;

            $file->save();
        }

        Toastr::success( __('student.student-update-message'), __('student.success'), ["positionClass" => "toast-top-center"]);
        return redirect()->route('student.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        $student = Student::find($id);

        Toastr::success( __('student.student-delete-message'), __('student.success'), ["positionClass" => "toast-top-center"]);
        $student->delete();
        return redirect()->route('student.index');
    }

    public function payments()
    {
        if( auth()->user()->hasRole('registrar')  ){
            $studentsIds = Student::where('college_id', auth()->user()->college_id)->pluck('student_id');
        } else {
            $studentsIds = Student::pluck('student_id');
        }

        $studentsPayments = collect();

        foreach ($studentsIds as $studentId) {
            $payment = Payment::where('student_id', $studentId)->first();
            if($payment != null){
                $studentsPayments->push($payment);
            }
        }

        if ($studentsPayments){
            foreach ($studentsPayments as $studentPayment) {
                $student = Student::find($studentPayment->student_id);

                if($student){
                    $student['semester'] = Semester::find($studentPayment->semester_id);

                    $fee = Registration::where(['student_id' => $student->student_id, 'payment_reference' => $studentPayment->payment_reference])->first();

                    if($fee) {
                        if($fee->payment_amount == $studentPayment->payment_amount){
                            $student['constraint_type'] = 'registered';
                        } elseif ($fee->payment_amount > $studentPayment->payment_amoun / 2) {
                            $student['constraint_type'] = 'partially-registered';
                        } else{
                            $student['constraint_type'] = 'un-registered';
                        }
                    } else{
                        $student['constraint_type'] = 'un-registered';
                    }

                    $studentPayment['college_id'] = $student->college_id;
                    $studentPayment['section_id'] = $student->section_id;
                    $studentPayment['program_id'] = $student->program_id;
                    $studentPayment['batch_id'] = $student->batch_id;

                    $studentPayment['student'] = $student;

                }
            }
        }

        $colleges = UserHelper::userColleges(auth()->user());

        return view('students.payments', [
            'colleges' => $colleges,
            'programs' => Program::all(),
            'batches' => Batch::all(),
            'studentsPayments' => $studentsPayments,
        ]);
    }

    public function filterStudentPayment(Request $request)
    {

        if(  auth()->user()->hasRole('registrar')  ){
            $studentsIds = Student::where('college_id', auth()->user()->college_id)->pluck('student_id');
        } else {
            $studentsIds = Student::pluck('student_id');
        }

        $studentsPayments = collect();

        foreach ($studentsIds as $studentId) {
            $payment = Payment::where('student_id', $studentId)->first();
            if($payment != null){
                $studentsPayments->push($payment);
            }
        }

        if ($studentsPayments){
            foreach ($studentsPayments as $studentPayment) {
                $student = Student::find($studentPayment->student_id);
                if($student){
                    $student['semester'] = Semester::find($studentPayment->semester_id);

                    $fee = Registration::where(['student_id' => $student->student_id, 'payment_reference' => $studentPayment->payment_reference])->first();

                    if($fee) {
                        if($fee->payment_amount == $studentPayment->payment_amount){
                            $student['constraint_type'] = 'registered';
                        } elseif ($fee->payment_amount > $studentPayment->payment_amoun / 2) {
                            $student['constraint_type'] = 'partially-registered';
                        } else{
                            $student['constraint_type'] = 'un-registered';
                        }
                    } else{
                        $student['constraint_type'] = 'un-registered';
                    }

                    $studentPayment['college_id'] = $student->college_id;
                    $studentPayment['section_id'] = $student->section_id;
                    $studentPayment['program_id'] = $student->program_id;
                    $studentPayment['batch_id'] = $student->batch_id;

                    $studentPayment['student'] = $student;

                }
            }

            if ($request->has('filter')) {
                if ($request->filled('college_id') && $request->college_id != 0) {
                    $studentsPayments = $studentsPayments->where('college_id', $request->college_id);
                }

                if ($request->filled('section_id') && $request->section_id != 0) {
                    $studentsPayments = $studentsPayments->where('section_id', $request->section_id);
                }

                if ($request->filled('program_id') && $request->program_id != 0) {
                    $studentsPayments = $studentsPayments->where('program_id', $request->program_id);
                }

                if ($request->filled('batch_id') && $request->batch_id != 0) {
                     $studentsPayments = $studentsPayments->where('batch_id', $request->batch_id);
                }
            }

            $colleges = UserHelper::userColleges(auth()->user());

            $sections = UserHelper::userSections($request->college_id);

            return view('students.payments', [
                'colleges' => $colleges,
//                'sections' => Section::where(['parent' => $request->filled('college_id') ? $request->college_id : 0])->get(),
                'sections' => $sections,
                'programs' => Program::all(),
                'batches' => Batch::all(),
                'semesters' => Semester::all(),

                'college_id' => $request->college_id,
                'section_id' => $request->section_id,
                'program_id' => $request->program_id,
                'batch_id' => $request->batch_id,

                'allStudentsPayments' => $studentsPayments,
                'studentsPayments' => $studentsPayments,
            ]);
        }
    }

    public function getCollegeSections(Request $request)
    {
        $sections = Section::where(['parent' => $request->parent_id])->get(['id', 'name']);

        return $sections;
    }

    // internal payment
    public function getStudentInfo(Request $request)
    {
        $student = Student::find($request->student_id);

        if(!$student){
            return response()->json('لم يتم العثور علي الطالب في السجلات', 404);
        }

        $result = Registration::where('student_id', $student->student_id)->latest()->first([
            'student_id', 'student_name', 'payment_amount', 'payment_reference', 'semester_id', 'is_paid','created_at'
        ]);

        if (!$result) {
            return response()->json('لم يتم اضافة دفعية للطالب!', 404);
        }

        $payment = Payment::where('payment_reference', $result->payment_reference)->first();

        $result['payment_status'] = $result->is_paid == 1 ? 'تم السداد' : 'لم يتم السداد';
        $result['semester_name'] = Semester::find($result->semester_id)->name ?? 'لم يتم التحديد';
        $result['createdAt'] = $payment ? date('Y-m-d', strtotime($payment->created_at)) : 'لم يتم السداد';
        $result['transaction_ref'] = $payment ? $payment->transaction_ref : 'لم يتم التحديد';
        $result['user_name'] = auth()->user()->name ?? 'لم يتم التحديد';

        return response()->json($result, 200);
    }

    public function payStudentPayment()
    {
        return view('students.pay-student-payment',[
            'semesters' => Semester::all(),
        ]);
    }

    public function payPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'payment_amount' => 'required',
            'payment_reference' => "required",
            'transaction_ref' => 'required',
            'semester_id' => 'required',
            'bank_identifier' => 'required',
            'bank_id' => 'nullable',
        ]);

        if ($validator->fails()) {
            Toastr::error("يوجد خطاء في الادخال", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $bankIdentifier = Bank::where('bank_identifier', $request->bank_identifier)->first();

        if(!$bankIdentifier){
            Toastr::error("هذا المستخدم غير مخول باجراء العملية", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $student = Student::where('student_id', $request->student_id)->first();

        if(!$student){
            Toastr::error("لم يتم ايجاد هذا الطالب في السجلات", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $semester = Semester::where('id', $request->semester_id)->first();

        if(!$semester){
            Toastr::error("لم يتم ايجاد السمستر المعني في السجلات", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $registration = Registration::where([ 'student_id' => $student->student_id, 'payment_reference' => $request->payment_reference, 'semester_id' => $semester->id ])->first();

        if(!$registration){
            Toastr::error("لم يتم ايجاد هذه الدفعية في السجلات", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $payment = new Payment();

        if($request->payment_amount < $registration->payment_amount){
            Toastr::error("مقدار الدفعية اقل من المبلغ المطلوب", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        } else if($request->payment_amount > $registration->payment_amount) {
            Toastr::error("مقدار الدفعية اكبر من المبلغ المطلوب", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        if($registration->is_paid == 1){
            Toastr::error("جميع دفعيات الطالب مسددة", __('student.error'), ["positionClass" => "toast-top-center"]);
            return redirect()->back();
        }

        $payment['student_id'] = $student->student_id;
        $payment['student_name'] = $student->student_name;

        $payment['payment_reference'] = $request->payment_reference;
        $payment['semester_id'] = $request->semester_id;
        $payment['payment_amount'] = $request->payment_amount;
        $payment['transaction_ref'] = $request->transaction_ref;

        $payment['bank_id'] = $bankIdentifier->id;
        $payment['fee_type_id'] = $registration->fee_type_id;

        // $payment['study_fees'] = $registration->study_fees / 2;
        $payment['study_fees'] = $registration->study_fees;
        $payment['registration_fees'] = $registration->registration_fees;
        $payment['insurance_fees'] = $registration->insurance_fees;
        $payment['discount'] = $registration->discount;

        $payment->save();

        $registration->is_paid = 1;
        $registration->save();

        Toastr::success( __('تم اضافة الدفعية بنجاح'), __('student.success'), ["positionClass" => "toast-top-center"]);
        return redirect()->back();
    }

    public function files($id)
    {
        $student = Student::where(['id' => $id])->first();

        if(!$student){
            return redirect()->back()->withErrors('not found')->withInput();
        }

        $files = $student->files;

        return view('students.files', [
            'student' => $student,
            'files' => $files,
            'fileTypes' => FileType::all(),
        ]);
    }

    public function excludeStudent(Request $request)
    {
        $student = Student::find($request->studentId);

        if(!$student){
            return response()->json("not-found", 404);
        }

        $student->excluded == 0 ? $student->excluded = 1 : $student->excluded = 0;

        $student->update();

        return response()->json($request, 200);
    }

    public function includedStudents(Request $request)
    {
        $students = Student::where('excluded', '!=', 1)->get();

        $students = StudentHelper::filterStudents($students, $request);

        return response()->json($students, 200);
    }

    public function getStateLocalities(Request $request)
    {
        $localities = Locality::where(['state_id' => $request->parent_id])->get(['id', 'name']);

        return $localities;
    }

    public function search(Request $request)
    {
        $keyword = $request->input('keyword');

        if (!empty($keyword)) {
            $students = Student::query()
                ->when($keyword, function ($query, $keyword) {
                    $query->where('student_number', 'like', "%{$keyword}%")
                        ->orWhere('student_name', 'like', "%{$keyword}%");
                })
                ->orderBy('id', 'desc')
                ->paginate(10);
        } else {
            $user = auth()->user();
            $filters = UserHelper::getRequestFilters();
            $collegeId = UserHelper::getCollegeId($user, $filters);
            $students = UserHelper::userStudents($user, $filters, true, 20);
        }

        return response()->json([
            'html' => view('students.partials.table', compact('students'))->render()
        ]);

    }
}
