<?php

namespace App\Models;

use App\Models\Bank;
use App\Models\Student;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Payment extends Model
{
    use SoftDeletes;
    use LogsActivity;

    Protected $fillable = [
        'id',
        'student_id',
        'student_name',
        'payment_reference',
        'semester_id',
        'payment_amount',
        'transaction_ref',
        'bank_id',
        'fee_type_id',
        'study_fees',
        'registration_fees',
        'insurance_fees',
        'discount',
        'penalty',
        'stamp',
        'created_at',
        'updated_at',
        'deleted_at'
    ];

    protected $casts = [
        'created_at' => 'datetime:Y-m-d H:i:s',
        'updated_at' => 'datetime:Y-m-d H:i:s',
    ];

    protected $appends = [
        'constraint_id'
    ];

    /**
     * Get the student that owns the Payment
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function student(): BelongsTo
    {
        return $this->belongsTo(Student::class, 'student_id');
    }

    /**
     * Get the semester that owns the Payment
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function semester(): BelongsTo
    {
        return $this->belongsTo(Semester::class);
    }

    /**
     * Get the bank that owns the Payment
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function bank(): BelongsTo
    {
        return $this->belongsTo(Bank::class);
    }

    public function getActivitylogOptions(): LogOptions
    {
        // Chain fluent methods for configuration options
        return LogOptions::defaults()
        ->logFillable()
        ->useLogName('payment');
        // ->logOnly(['student_id', 'student_name']);
    }

    /**
     * Get the registration that owns the Payment
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function registration(): BelongsTo
    {
        return $this->belongsTo(Registration::class, 'payment_reference', 'payment_reference');
    }

    public function getConstraintIdAttribute(): int
    {
        $feeAmount = $this->registration->fees_total ?? 0;
        $paymentAmount = $this->payment_amount;

        if ($feeAmount != 0 && $feeAmount <= $paymentAmount) {
            return 1; // Fully registered
        }

        if ($paymentAmount >= ($feeAmount / 2)) {
            return 2; // Partially registered
        }

        return 3; // Unregistered
    }
}
