<?php

namespace App\Models;

use App\Models\Fee;
use App\Models\File;
use App\Models\Batch;
use App\Models\Image;
use App\Models\College;
use App\Models\Payment;
use App\Models\Section;
use App\Models\Scholarship;
use App\Models\AcademicYear;
use App\Models\Registration;
use App\Models\EducationStatus;
use Spatie\Activitylog\LogOptions;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\Traits\LogsActivity;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Student extends Model
{
    use HasFactory;
    use SoftDeletes;
    use LogsActivity;

    protected $primaryKey = 'student_id';

    public $incrementing = false;

    protected $keyType = 'string';

    protected $with = [
        'section', 'program'
    ];

    Protected $fillable = [
        'student_id',
        'student_number',
        'student_name',
        'admission_degree',
        'gender',
        'program_fees',
        'registration_fees',
        'insurance_fees',
        'discount',
        'date_of_birth',
        'phone_number',
        'phone_number2',
        'email',
        'nationality',
        'guardian_phone_number',
        'national_id',
        'college_id',
        'program_id',
        'section_id',
        'admission_id',
        'batch_id',
        'addmission_type',
        'addmission_date',
        'addmission_form_no',
        'certificate_type',
        'notes',
        'excluded',

        'accommodation_type',
        'acceptance_rate',
        'guardian_name',
        'guardian_address',
        'state_id',
        'locality_id',
    ];

    /**
     * The feeTypes that belong to the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function fees(): BelongsToMany
    {
        return $this->belongsToMany(Fee::class, 'student_fees', 'student_id', 'fee_code')
            ->withPivot('amount', 'paid', 'penalty', 'note', 'payment_status')
            ->withTimestamps();
    }

    /**
     * The academicYears that belong to the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function academicYears(): BelongsToMany
    {
        return $this->belongsToMany(AcademicYear::class, 'student_academic_years', 'student_id');
    }

    /**
     * Get the section that owns the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function section(): BelongsTo
    {
        return $this->belongsTo(Section::class);
    }

    /**
     * Get the program that owns the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function program(): BelongsTo
    {
        return $this->belongsTo(Program::class);
    }

    /**
     * Get all of the student's images.
     */
     public function images()
    {
        return $this->morphMany(Image::class, 'imageable');
    }

    /**
     * Get all of the student's files.
     */
    public function files()
    {
        return $this->morphMany(File::class, 'fileable')->orderBy('created_at', 'desc');
    }

    /**
     * The scholarships that belong to the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function scholarships(): BelongsToMany
    {
        return $this->belongsToMany(Scholarship::class, 'student_scholar_ships', 'student_id', 'scholarship_id');
    }

    /**
     * The educationStatuses that belong to the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function educationStatuses(): BelongsToMany
    {
        return $this->belongsToMany(EducationStatus::class, 'student_education_statuses', 'student_id', 'education_status_id');
    }

    /**
     * Get the college that owns the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function college(): BelongsTo
    {
        return $this->belongsTo(College::class, 'college_id');
    }

    /**
     * Get the batch that owns the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function batch(): BelongsTo
    {
        return $this->belongsTo(Batch::class, 'batch_id');
    }

    /**
     * Get all of the payments for the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class, 'student_id')->orderBy('created_at', 'desc');
    }

    /**
     * Get all of the registrations for the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function registrations(): HasMany
    {
        return $this->hasMany(Registration::class, 'student_id')->orderBy('created_at', 'desc');
    }

    /**
     * Get the card associated with the Student
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function card(): HasOne
    {
        return $this->hasOne(StudentCard::class, 'student_id');
    }

    public function getActivitylogOptions(): LogOptions
    {
        // Chain fluent methods for configuration options
        return LogOptions::defaults()
        ->logFillable()
        ->useLogName('student');
        // ->logOnly(['student_id', 'student_name']);
    }
}
